// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.audio;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.zip.ZipException;

import org.refcodes.tabular.CsvStringRecordReader;
import org.refcodes.tabular.Record;

/**
 * The {@link CsvMonoSampleReader} provides means to read sound samples from a
 * CSV file.
 */
public class CsvMonoSampleReader extends AbstractCsvSampleReader<MonoSample> implements MonoSampleReader {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private MonoSampleBuilderImpl _soundSample = new MonoSampleBuilderImpl( (long) 0, SamplingRate.AUDIO_CD.getSamplesPerSecond() );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link CsvMonoSampleReader} for writing sound samples to a
	 * CSV file or stream.
	 * 
	 * @param aFile The {@link File} where to write the CSV records to.
	 * 
	 * @throws IOException thrown in case there was an I/O related problem.
	 * @throws ZipException Signals that a Zip exception of some sort has
	 *         occurred.
	 */
	public CsvMonoSampleReader( File aFile ) throws ZipException, IOException {
		super( aFile );
	}

	/**
	 * Constructs the {@link CsvMonoSampleReader} for writing sound samples to a
	 * CSV file or stream.
	 * 
	 * @param aInputStream The {@link InputStream} where to write the CSV
	 *        records to.
	 *
	 * @throws IOException thrown in case there was an I/O related problem.
	 */
	public CsvMonoSampleReader( InputStream aInputStream ) throws IOException {
		super( aInputStream );
	}

	/**
	 * Constructs the {@link CsvMonoSampleReader} for writing sound samples to a
	 * CSV file or stream.
	 * 
	 * @param aCsvReader The {@link CsvStringRecordReader} with which to write
	 *        the CSV records with.
	 */
	protected CsvMonoSampleReader( CsvStringRecordReader aCsvReader ) {
		super( aCsvReader );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MonoSample nextRow() throws IOException {
		doProbeHeader();
		Record<String> theSampleCsv = _csvReader.next();
		String eSampleDataCsv;
		double theSamplingData = 0;
		if ( theSampleCsv.containsKey( CsvSoundSampleWriter.HEADER_SAMPLE_DATA ) ) {
			eSampleDataCsv = theSampleCsv.get( CsvSoundSampleWriter.HEADER_SAMPLE_DATA );
			if ( eSampleDataCsv != null ) {
				theSamplingData = CsvSoundSampleWriter.toDouble( eSampleDataCsv );
			}
		}

		_soundSample.setMonoData( theSamplingData );

		String theIndex = theSampleCsv.get( CsvSoundSampleWriter.HEADER_INDEX );
		if ( theIndex != null && theIndex.length() != 0 ) {
			_soundSample.setIndex( Long.parseLong( theIndex ) );
		}
		else {
			_soundSample.increaseIndex();
		}

		String theSamplingRate = theSampleCsv.get( CsvSoundSampleWriter.HEADER_SAMPLING_RATE );
		if ( theSamplingRate != null && theSamplingRate.length() != 0 ) {
			_soundSample.setSamplingRate( Integer.parseInt( theSamplingRate ) );
		}

		String theTimeStamp = theSampleCsv.get( CsvSoundSampleWriter.HEADER_TIME_STAMP );
		if ( theTimeStamp != null && theTimeStamp.length() != 0 ) {
			BigDecimal theDecimal = new BigDecimal( theTimeStamp );
			_soundSample.setTimeStamp( theDecimal.doubleValue() );
		}
		else {
			_soundSample.updateTimeStamp();
		}
		return new MonoSampleImpl( _soundSample );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String nextRaw() {
		return _csvReader.nextRaw();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double nextMonoData() throws IOException {
		return nextRow().getMonoData();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSamplingRate() {
		return _soundSample.getSamplingRate();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long getIndex() {
		return _soundSample.getIndex();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
