// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.audio;

import java.util.Arrays;

/**
 * A {@link SoundSampleBuilder} represents multiple amplitudes (one for each
 * channel,e.g. two in case we have classical stereo sound) assigned to an
 * according time positioning.
 */
public class SoundSampleBuilderImpl implements SoundSampleBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private double _timeStamp = -1;
	private int _sampligRate = -1;
	private double[] _sampleData = null;
	private long _index = -1;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link SoundSampleImpl} instance with the according sound
	 * sample's settings
	 * 
	 * @param aSoundSample The sound sample from which to create a new instance.
	 */
	public SoundSampleBuilderImpl( SoundSample aSoundSample ) {
		_index = aSoundSample.getIndex();
		_timeStamp = aSoundSample.getTimeStamp();
		_sampligRate = aSoundSample.getSamplingRate();
		if ( aSoundSample.getSampleData() != null ) {
			_sampleData = Arrays.copyOf( aSoundSample.getSampleData(), aSoundSample.getChannelCount() );
		}
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 */
	public SoundSampleBuilderImpl( double[] aSampleData ) {
		_sampleData = aSampleData;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 */
	public SoundSampleBuilderImpl( double aTimeMillis, double[] aSampleData ) {
		_timeStamp = aTimeMillis;
		_sampleData = aSampleData;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleBuilderImpl( double aTimeMillis, double[] aSampleData, int aSamplingRate ) {
		_timeStamp = aTimeMillis;
		_sampleData = aSampleData;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleBuilderImpl( double[] aSampleData, int aSamplingRate ) {
		_sampleData = aSampleData;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 */
	public SoundSampleBuilderImpl( long aIndex, double[] aSampleData ) {
		_index = aIndex;
		_sampleData = aSampleData;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleBuilderImpl( long aIndex, double[] aSampleData, int aSamplingRate ) {
		_index = aIndex;
		_sampleData = aSampleData;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 */
	public SoundSampleBuilderImpl( long aIndex, double aTimeMillis, double[] aSampleData ) {
		_index = aIndex;
		_timeStamp = aTimeMillis;
		_sampleData = aSampleData;
	}

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSampleData The sample (set of values representing the channels,
	 *        one channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleBuilderImpl( long aIndex, double aTimeMillis, double[] aSampleData, int aSamplingRate ) {
		_index = aIndex;
		_timeStamp = aTimeMillis;
		_sampleData = aSampleData;
		_sampligRate = aSamplingRate;
	}

	// -------------------------------------------------------------------------

	/**
	 * Constructs a {@link SoundSampleBuilder} instance with the according time
	 * and sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleBuilderImpl( long aIndex, int aSamplingRate ) {
		_index = aIndex;
		_sampligRate = aSamplingRate;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double[] getSampleData() {
		return _sampleData;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double getTimeStamp() {
		return _timeStamp;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSamplingRate() {
		return _sampligRate;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long getIndex() {
		return _index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIndex( long aIndex ) {
		_index = aIndex;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void increaseIndex() {
		_index++;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void decreaseIndex() {
		_index--;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSampleData( double[] aSampleData ) {
		_sampleData = aSampleData;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTimeStamp( double aTimeInMillis ) {
		_timeStamp = aTimeInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSamplingRate( int aSamplingRate ) {
		_sampligRate = aSamplingRate;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "SoundSampleBuilderImpl [index=" + getIndex() + ", timeStamp=" + getTimeStamp() + ", sampleData=" + Arrays.toString( getSampleData() ) + ", sampligRate=" + getSamplingRate() + "]";
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
