// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.audio;

import java.util.Arrays;

/**
 * A {@link SoundSample} represents multiple amplitudes (one for each
 * channel,e.g. two in case we have classical stereo sound) assigned to an
 * according time positioning.
 */
public class SoundSampleImpl implements SoundSample {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private double _timeStamp = -1;
	private int _sampligRate = -1;
	private double[] _sampleData = null;
	private long _index = -1;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link SoundSample} instance with the according sound
	 * sample's settings
	 * 
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 */
	public SoundSampleImpl( SoundSample aSample ) {
		_index = aSample.getIndex();
		_timeStamp = aSample.getTimeStamp();
		_sampligRate = aSample.getSamplingRate();
		if ( aSample.getSampleData() != null ) {
			_sampleData = Arrays.copyOf( aSample.getSampleData(), aSample.getChannelCount() );
		}
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 */
	public SoundSampleImpl( double[] aSample ) {
		_sampleData = aSample;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 */
	public SoundSampleImpl( double aTimeMillis, double[] aSample ) {
		_timeStamp = aTimeMillis;
		_sampleData = aSample;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleImpl( double aTimeMillis, double[] aSample, int aSamplingRate ) {
		_timeStamp = aTimeMillis;
		_sampleData = aSample;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleImpl( double[] aSample, int aSamplingRate ) {
		_sampleData = aSample;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 */
	public SoundSampleImpl( long aIndex, double[] aSample ) {
		_index = aIndex;
		_sampleData = aSample;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleImpl( long aIndex, double[] aSample, int aSamplingRate ) {
		_index = aIndex;
		_sampleData = aSample;
		_sampligRate = aSamplingRate;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 */
	public SoundSampleImpl( long aIndex, double aTimeMillis, double[] aSample ) {
		_index = aIndex;
		_timeStamp = aTimeMillis;
		_sampleData = aSample;
	}

	/**
	 * Constructs a {@link SoundSample} instance with the according time and
	 * sample settings.
	 * 
	 * @param aIndex The index of the sample according to its position in the
	 *        sample sequence (e.g. sound file).
	 * @param aTimeMillis The time positioning of this sound sample.
	 * @param aSample The sample (set of values representing the channels, one
	 *        channel per value) for the related time positioning.
	 * @param aSamplingRate The sampling rate for the given sample.
	 */
	public SoundSampleImpl( long aIndex, double aTimeMillis, double[] aSample, int aSamplingRate ) {
		_index = aIndex;
		_timeStamp = aTimeMillis;
		_sampleData = aSample;
		_sampligRate = aSamplingRate;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double[] getSampleData() {
		return _sampleData;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double getTimeStamp() {
		return _timeStamp;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSamplingRate() {
		return _sampligRate;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long getIndex() {
		return _index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "SoundSampleImpl [index=" + getIndex() + ", timeStamp=" + getTimeStamp() + ", sampleData=" + Arrays.toString( getSampleData() ) + ", sampligRate=" + getSamplingRate() + "]";
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
