package org.refcodes.checkerboard.impls;

import org.refcodes.checkerboard.Checkerboard;
import org.refcodes.checkerboard.CheckerboardViewer;
import org.refcodes.checkerboard.Player;
import org.refcodes.checkerboard.SpriteFactory;
import org.refcodes.checkerboard.ViewportOffsetChangedEvent;
import org.refcodes.component.InitializeException;
import org.refcodes.graphical.Dimension;
import org.refcodes.graphical.GridMode;
import org.refcodes.graphical.Offset;
import org.refcodes.graphical.Position;
import org.refcodes.graphical.ViewportDimension;
import org.refcodes.graphical.ViewportOffset;
import org.refcodes.graphical.impls.ViewportDimensionImpl.ViewportDimensionPropertyBuilderImpl;
import org.refcodes.observer.SubscribeEvent;
import org.refcodes.observer.UnsubscribeEvent;

/**
 * In order to provide a {@link Checkerboard}, register an observer by invoking
 * {@link Checkerboard#subscribeObserver(Object)}. The default
 * {@link CheckerboardImpl#subscribeObserver(org.refcodes.checkerboard.CheckerboardObserver)}
 * method will invoke this {@link #onSubscribe(SubscribeEvent)} method which in
 * turn sets the {@link Checkerboard} instance. Retrieve it by calling
 * {@link #getCheckerboard()}
 *
 * @param <S> The type which's instances represent a {@link Player} state.
 * @param <IMG> The type for the state's representation ("image").
 * @param <CBV> The {@link CheckerboardViewer}'s type implementing this
 *        interface.
 */
public abstract class AbstractCheckerboardViewer<S, IMG, SF extends SpriteFactory<IMG, S, ? extends CBV>, CBV extends CheckerboardViewer<S, CBV>> implements CheckerboardViewer<S, CBV> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private int _viewportWidth = -1;
	private int _viewportHeight = -1;
	private int _viewportOffsetX = 0;
	private int _viewportOffsetY = 0;
	private ViewportDimensionPropertyBuilderBuilder _minViewportDimension = new ViewportDimensionPropertyBuilderImpl( -1, -1 );
	private Checkerboard<S> _checkerboard;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@SuppressWarnings("unchecked")
	@Override
	public CBV withInitialize() throws InitializeException {
		initialize();
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffsetY( int aPosY ) {
		setViewportOffsetY( aPosY );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportHeight( int aGridHeight ) {
		setViewportHeight( aGridHeight );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportWidth( int aGridWidth ) {
		setViewportWidth( aGridWidth );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportDimension( int aWidth, int aHeight ) {
		setViewportDimension( aWidth, aHeight );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportDimension( ViewportDimension aGridDimension ) {
		setViewportDimension( aGridDimension );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportDimension( Dimension aDimension ) {
		setViewportDimension( aDimension );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffset( int aPosX, int aPosY ) {
		setViewportOffset( aPosX, aPosY );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffset( Position aOffset ) {
		setViewportOffset( aOffset );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffset( ViewportOffset aOffset ) {
		setViewportOffset( aOffset );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffset( Offset aOffset ) {
		setViewportOffset( aOffset );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withViewportOffsetX( int aPosX ) {
		setViewportOffsetX( aPosX );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withMinViewportDimension( ViewportDimension aDimension ) {
		setMinViewportDimension( aDimension );
		return (CBV) this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CBV withMinViewportDimension( int aWidth, int aHeight ) {
		setMinViewportDimension( aWidth, aHeight );
		return (CBV) this;
	}

	@Override
	public void setViewportDimension( int aWidth, int aHeight ) {
		if ( _minViewportDimension.getViewportWidth() != -1 && aWidth < _minViewportDimension.getViewportWidth() ) { throw new IllegalArgumentException( "The provided grid width <" + aWidth + "> is less than the min grid width (<" + _minViewportDimension.getViewportWidth() + ">)." ); }
		if ( _minViewportDimension.getViewportHeight() != -1 && aHeight < _minViewportDimension.getViewportHeight() ) { throw new IllegalArgumentException( "The provided grid height <" + aHeight + "> is less than the min grid height (<" + _minViewportDimension.getViewportHeight() + ">)." ); }
		_viewportWidth = aWidth;
		_viewportHeight = aHeight;
	}

	@Override
	public void setViewportDimension( ViewportDimension aGridDimension ) {
		setViewportDimension( aGridDimension.getViewportWidth(), aGridDimension.getViewportHeight() );
	}

	@Override
	public void setViewportDimension( Dimension aDimension ) {
		setViewportDimension( aDimension.getWidth(), aDimension.getHeight() );
	}

	@Override
	public void setViewportWidth( int aGridWidth ) {
		setViewportDimension( aGridWidth, _viewportHeight );
	}

	@Override
	public void setViewportHeight( int aGridHeight ) {
		setViewportDimension( _viewportWidth, aGridHeight );
	}

	@Override
	public int getViewportWidth() {
		return _viewportWidth;
	}

	@Override
	public int getViewportHeight() {
		return _viewportHeight;
	}

	@Override
	public void setViewportOffset( int aPosX, int aPosY ) {
		if ( aPosX != _viewportOffsetX || aPosY != _viewportOffsetY ) {
			ViewportOffsetChangedEvent<S> theEvent = new ViewportOffsetChangedEventImpl<S>( aPosX, aPosY, _viewportOffsetX, _viewportOffsetY, this );
			_viewportOffsetX = aPosX;
			_viewportOffsetY = aPosY;
			onViewportOffsetChangedEvent( theEvent );
		}
	}

	@Override
	public void setViewportOffset( Position aOffset ) {
		setViewportOffset( aOffset.getPositionX(), aOffset.getPositionY() );
	}

	@Override
	public void setViewportOffset( ViewportOffset aOffset ) {
		setViewportOffset( aOffset.getViewportOffsetX(), aOffset.getViewportOffsetY() );
	}

	@Override
	public void setViewportOffset( Offset aOffset ) {
		setViewportOffset( aOffset.getOffsetX(), aOffset.getOffsetY() );
	}

	@Override
	public void setViewportOffsetX( int aPosX ) {
		setViewportOffset( aPosX, _viewportOffsetY );
	}

	@Override
	public int getViewportOffsetX() {
		return _viewportOffsetX;
	}

	@Override
	public int getViewportOffsetY() {
		return _viewportOffsetY;
	}

	@Override
	public void setViewportOffsetY( int aPosY ) {
		setViewportOffset( _viewportOffsetX, aPosY );
	}

	@Override
	public void setMinViewportDimension( ViewportDimension aDimension ) {
		_minViewportDimension.setViewportDimension( aDimension );
	}

	@Override
	public ViewportDimension getMinViewportDimension() {
		return _minViewportDimension;
	}

	@Override
	public void setMinViewportDimension( int aWidth, int aHeight ) {
		_minViewportDimension.setViewportDimension( aWidth, aHeight );
	}

	@Override
	public GridMode getGridMode() {
		return getCheckerboard().getGridMode();
	}

	@Override
	public int getGridWidth() {
		return getCheckerboard().getGridWidth();
	}

	@Override
	public int getGridHeight() {
		return getCheckerboard().getGridHeight();
	}

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void destroy() {}

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void onSubscribe( SubscribeEvent<Checkerboard<S>> aSubscribeEvent ) {
		_checkerboard = aSubscribeEvent.getSource();
	}

	@Override
	public void onUnsubscribe( UnsubscribeEvent<Checkerboard<S>> aUnsubscribeEvent ) {}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	protected Checkerboard<S> getCheckerboard() {
		if ( _checkerboard == null ) { throw new IllegalStateException( "Register your checkerboard viewer to a checkerboard via \"Checkerboard#subscribeObserver( yourCheckerboardViewer)\". The viewer's \"#onSubscribe( ... )} method will be invoked, retrieving (from the event) and setting the checkerboard object to be returnd by this \"getCheckerboard()\" method." ); }
		return _checkerboard;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
