package org.refcodes.checkerboard.impls;

import org.refcodes.checkerboard.ChangePositionEvent;
import org.refcodes.checkerboard.DraggabilityChangedEvent;
import org.refcodes.checkerboard.Player;
import org.refcodes.checkerboard.PlayerEvent;
import org.refcodes.checkerboard.PlayerObserver;
import org.refcodes.checkerboard.PositionChangedEvent;
import org.refcodes.checkerboard.StateChangedEvent;
import org.refcodes.checkerboard.VisibilityChangedEvent;
import org.refcodes.controlflow.ExecutionStrategy;
import org.refcodes.exception.VetoException;
import org.refcodes.exception.VetoException.VetoRuntimeException;
import org.refcodes.graphical.Position;
import org.refcodes.observer.impls.AbstractObservable;

public class PlayerImpl<S> extends AbstractObservable<PlayerObserver<S>, PlayerEvent<S>>implements Player<S> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final ExecutionStrategy STRATEGY = ExecutionStrategy.SEQUENTIAL;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private int _posX;
	private int _posY;
	private S _state;
	private boolean _isVisible = true;
	private boolean _isDraggable = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	public PlayerImpl( int aPosX, int aPosY ) {
		_posX = aPosX;
		_posY = aPosY;
	}

	public PlayerImpl( Position aPosition ) {
		_posX = aPosition.getPositionX();
		_posY = aPosition.getPositionY();
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public Player<S> withPosition( int aPosX, int aPosY ) throws VetoRuntimeException {
		setPosition( aPosX, aPosY );
		return this;
	}

	@Override
	public Player<S> withPosition( Position aPosition ) throws VetoRuntimeException {
		setPosition( aPosition );
		return this;
	}

	@Override
	public void setPosition( int aPosX, int aPosY ) throws VetoRuntimeException {
		if ( aPosX != _posX || aPosY != _posY ) {
			ChangePositionEvent<S> theVetoable = new ChangePositionEventImpl<S>( aPosX, aPosY, _posX, _posY, this );
			try {
				fireEvent( theVetoable, STRATEGY );
			}
			catch ( VetoException e ) {
				throw new VetoRuntimeException( e );
			}
			PositionChangedEvent<S> theEvent = new PositionChangedEventImpl<S>( aPosX, aPosY, _posX, _posY, this );
			_posX = aPosX;
			_posY = aPosY;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public void setPosition( Position aPosition ) throws VetoRuntimeException {
		setPosition( aPosition.getPositionX(), aPosition.getPositionY() );

	}

	@Override
	public Player<S> withPositionY( int aPosY ) throws VetoRuntimeException {
		setPositionY( aPosY );
		return this;
	}

	@Override
	public void setPositionY( int aPosY ) throws VetoRuntimeException {

		if ( aPosY != _posY ) {
			PositionChangedEvent<S> theEvent = new PositionChangedEventImpl<S>( _posX, aPosY, _posX, _posY, this );
			_posY = aPosY;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public Player<S> withPositionX( int aPosX ) throws VetoRuntimeException {
		setPositionX( aPosX );
		return this;
	}

	@Override
	public void setState( S aState ) {

		if ( aState != _state ) {
			StateChangedEvent<S> theEvent = new StateChangedEventImpl<S>( aState, _state, this );
			_state = aState;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public Player<S> withState( S aState ) {
		setState( aState );
		return this;
	}

	@Override
	public void setPositionX( int aPosX ) {
		if ( aPosX != _posX ) {
			PositionChangedEvent<S> theEvent = new PositionChangedEventImpl<S>( aPosX, _posY, _posX, _posY, this );
			_posX = aPosX;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public void setVisible( boolean isVisible ) {
		if ( isVisible != _isVisible ) {
			VisibilityChangedEvent<S> theEvent = new VisibilityChangedEventImpl<S>( this );
			_isVisible = isVisible;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public Player<S> withShow() {
		setVisible( true );
		return this;
	}

	@Override
	public Player<S> withHide() {
		setVisible( false );
		return this;
	}

	@Override
	public Player<S> withVisible( boolean isVisible ) {
		setVisible( isVisible );
		return this;
	}

	@Override
	public void show() {
		setVisible( true );
	}

	@Override
	public void hide() {
		setVisible( false );
	}

	@Override
	public boolean isVisible() {
		return _isVisible;
	}

	@Override
	public void setDraggable( boolean isDraggable ) {
		if ( isDraggable != _isDraggable ) {
			DraggabilityChangedEvent<S> theEvent = new DraggabilityChangedEventImpl<S>( this );
			_isDraggable = isDraggable;
			try {
				fireEvent( theEvent, STRATEGY );
			}
			catch ( VetoException ignore ) {}
		}
	}

	@Override
	public Player<S> withDraggable() {
		setDraggable( true );
		return this;
	}

	@Override
	public Player<S> withStationary() {
		setDraggable( false );
		return this;
	}

	@Override
	public Player<S> withDraggable( boolean isDraggable ) {
		setDraggable( isDraggable );
		return this;
	}

	@Override
	public void draggable() {
		setDraggable( true );
	}

	@Override
	public void stationary() {
		setDraggable( false );
	}

	@Override
	public boolean isDraggable() {
		return _isDraggable;
	}

	@Override
	public int getPositionX() {
		return _posX;
	}

	@Override
	public int getPositionY() {
		return _posY;
	}

	@Override
	public S getState() {
		return _state;
	}

	@Override
	public String toString() {
		return getClass().getSimpleName() + "(" + _posX + " x " + _posY + ", " + getState() + ")@" + hashCode();

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	protected boolean fireEvent( PlayerEvent<S> aEvent, PlayerObserver<S> aObserver, ExecutionStrategy aExecutionStrategy ) throws VetoException {
		aObserver.onPlayerEvent( aEvent );
		if ( aEvent instanceof ChangePositionEvent ) {
			aObserver.onChangePositionEvent( (ChangePositionEvent<S>) aEvent );
		}
		if ( aEvent instanceof PositionChangedEvent ) {
			aObserver.onPositionChangedEvent( (PositionChangedEvent<S>) aEvent );
		}
		if ( aEvent instanceof StateChangedEvent ) {
			aObserver.onStateChangedEvent( (StateChangedEvent<S>) aEvent );
		}
		if ( aEvent instanceof VisibilityChangedEvent ) {
			aObserver.onVisibilityChangedEvent( (VisibilityChangedEvent<S>) aEvent );
		}
		if ( aEvent instanceof DraggabilityChangedEvent ) {
			aObserver.onDraggabilityChangedEvent( (DraggabilityChangedEvent<S>) aEvent );
		}
		return true;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
