// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.checkerboard;

import org.refcodes.mixin.ColumnWidthAccessor.ColumnWidthBuilder;
import org.refcodes.mixin.ColumnWidthAccessor.ColumnWidthProperty;

/**
 * Extends the interface {@link CheckerboardViewer} with functionality required
 * for console output.
 *
 * @param <P> The type representing a {@link Player}
 * @param <S> The type which's instances represent a {@link Player} state.
 */
public interface ConsoleCheckerboardViewer<P extends Player<P, S>, S> extends CheckerboardViewer<P, S, ConsoleCheckerboardViewer<P, S>>, ColumnWidthProperty, ColumnWidthBuilder<ConsoleCheckerboardViewer<P, S>> {

	/**
	 * Retrieves the refresh loop time between two redraw cycles of the
	 * {@link Checkerboard}. This is the delay time between succeeding print
	 * outs of the {@link Checkerboard} states to the console. If the refresh
	 * loop time is &gt; 0, then the {@link Checkerboard} is redrawn at most
	 * with intervals of the given refresh loop time. Attention: The
	 * {@link Checkerboard} is only redrawn in case the {@link Checkerboard}
	 * changed compared to the last redraw process.
	 * 
	 * @return The according time in milliseconds.
	 */
	int getRedrawLoopTimeInMillis();

	/**
	 * Determines whether the {@link Checkerboard} is printed out upon any
	 * according events from the underlying {@link Checkerboard} or as of the
	 * refresh loop time (see {@link #getRedrawLoopTimeInMillis()}). Attention:
	 * The {@link Checkerboard} is only redrawn in case the {@link Checkerboard}
	 * changed compared to the last redraw process.
	 * 
	 * @return True in case the {@link Checkerboard} is redrawn upon an event
	 *         from the {@link Checkerboard} or upon the refresh loop time.
	 */
	boolean isRedrawOnEvent();

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ConsoleCheckerboardViewer<P, S> withColumnWidth( int aColumnWidth ) {
		setColumnWidth( aColumnWidth );
		return this;
	}

}
