// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.collection;

import java.util.Collection;
import java.util.Set;

/**
 * This interface provides functionality for working with keys.
 *
 * @param <K> The type of the key.
 * @param <V> The type of the value which relates to a key.
 */
public interface Keys<K, V> {

	/**
	 * Tests whether there is an element with the given key.
	 * 
	 * @param aKey The key for which to test whether there is an element.
	 * 
	 * @return True in case there is an element for the given key.
	 */
	boolean containsKey( K aKey );

	/**
	 * Retrieves the element assigned to the given key.
	 * 
	 * @param aKey The key for which to get the element.
	 * 
	 * @return The column in question or null if there is none such column.
	 */
	V get( K aKey );

	/**
	 * Retrieves a collection containing all the keys found in the elements of
	 * this collection.
	 * 
	 * @return A collection with key objects being the keys of all elements in
	 *         this collection.
	 */
	Set<K> keySet();

	/**
	 * Returns a {@link Collection} view of the values related to the contained
	 * keys.
	 * 
	 * @return A {@link Collection} view of the values related to the contained
	 *         keys.
	 */
	Collection<V> values();

	/**
	 * Extends the {@link Keys} interface with {@link Clearable} functionality
	 * (as of {@link #clear()}). The case of having a plain {@link Keys} without
	 * dedicated {@link MutableKeys#removeKey(Object)} or
	 * {@link MutableKeys#put(Object, Object)} methods but with a dedicated
	 * {@link #clear()} method seems to be quite common, therefore this
	 * interface has been provided.
	 * 
	 * @param <K> The type of the key.
	 * @param <V> The type of the value which relates to a key.
	 */
	public interface ClearableKeys<K, V> extends Keys<K, V>, Clearable {}

	/**
	 * Adds mutable functionality to the {@link Keys} interface.
	 * 
	 * @param <K> The type of the key.
	 * @param <V> The type of the value which relates to a key.
	 */
	public interface MutableKeys<K, V> extends Keys<K, V>, Clearable {

		/**
		 * Removes an element identified by the given key.
		 * 
		 * @param aKey The key which's element is to be removed.
		 * 
		 * @return The value being removed.
		 */
		V removeKey( K aKey );
	}
}
