// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.refcodes.component.OpenException;
import org.refcodes.configuration.ResourceProperties.ResourcePropertiesBuilder;
import org.refcodes.data.Delimiter;
import org.refcodes.data.Scheme;
import org.refcodes.runtime.RuntimeUtility;

/**
 * Implementation of the mutable {@link ResourcePropertiesBuilder} interface.
 */
abstract public class AbstractResourcePropertiesBuilder extends PropertiesBuilderImpl implements ResourcePropertiesBuilder {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected File _propertiesFile;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Create an empty {@link AbstractResourcePropertiesBuilder} instance using
	 * the default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations.
	 */
	protected AbstractResourcePropertiesBuilder() {
		super();
	}

	/**
	 * Create a {@link AbstractResourcePropertiesBuilder} instance containing
	 * the elements as of {@link MutablePathMap#insert(Object)} using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations:
	 * 
	 * "Inspects the given object and adds all elements found in the given
	 * object. Elements of type {@link Map}, {@link Collection} and arrays are
	 * identified and handled as of their type: The path for each value in a
	 * {@link Map} is appended with its according key. The path for each value
	 * in a {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aObj The object from which the elements are to be added.
	 */
	protected AbstractResourcePropertiesBuilder( Object aObj ) {
		super( aObj );
	}

	/**
	 * Create a {@link AbstractResourcePropertiesBuilder} instance containing
	 * the elements of the provided {@link Properties} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	protected AbstractResourcePropertiesBuilder( Properties aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link AbstractResourcePropertiesBuilder} instance containing
	 * the elements of the provided {@link PropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	protected AbstractResourcePropertiesBuilder( PropertiesBuilder aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link AbstractResourcePropertiesBuilder} instance containing
	 * the elements of the provided {@link Map} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	protected AbstractResourcePropertiesBuilder( Map<?, ?> aProperties ) {
		super( aProperties );
	}

	/**
	 * Loads the properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( File aFile ) throws IOException, ParseException {
		loadFrom( aFile );
	}

	/**
	 * Loads the properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( File aFile, char... aDelimiters ) throws IOException, ParseException {
		loadFrom( aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( File aFile, boolean isSeekFile ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aFile );
		}
		else {
			loadFrom( aFile );
		}
	}

	/**
	 * Loads or seeks the properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( File aFile, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aFile, aDelimiters );
		}
		else {
			loadFrom( aFile, aDelimiters );
		}
	}

	/**
	 * Loads the properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public AbstractResourcePropertiesBuilder( String aFilePath ) throws IOException, ParseException {
		loadFrom( aFilePath );
	}

	/**
	 * Loads the properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		loadFrom( aFilePath, aDelimiters );
	}

	/**
	 * Loads the properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		loadFrom( aResourceClass, aFilePath );
	}

	/**
	 * Loads the properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		loadFrom( aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aFilePath );
		}
		else {
			loadFrom( aFilePath );
		}
	}

	/**
	 * Loads the properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aFilePath, aDelimiters );
		}
		else {
			loadFrom( aFilePath, aDelimiters );
		}
	}

	/**
	 * Loads the properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aResourceClass, aFilePath );
		}
		else {
			loadFrom( aResourceClass, aFilePath );
		}
	}

	/**
	 * Loads the properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument, then the properties are, if the
	 * file does not exist, loaded from the first folder containing such a file
	 * as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		if ( isSeekFile ) {
			seekFrom( aResourceClass, aFilePath, aDelimiters );
		}
		else {
			loadFrom( aResourceClass, aFilePath, aDelimiters );
		}
	}

	/**
	 * Loads the properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( URL aUrl ) throws IOException, ParseException {
		loadFrom( aUrl );
	}

	/**
	 * Loads the properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		loadFrom( aUrl, aDelimiters );
	}

	/**
	 * Reads the properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( InputStream aInputStream ) throws IOException, ParseException {
		loadFrom( aInputStream );
	}

	/**
	 * Reads the properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	protected AbstractResourcePropertiesBuilder( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		loadFrom( aInputStream, aDelimiters );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties loadFrom( File aFile, char... aDelimiters ) throws IOException, ParseException {
		try (InputStream theInputStream = new FileInputStream( aFile )) {
			Properties theResult = loadFrom( theInputStream, aDelimiters );
			_propertiesFile = aFile;
			return theResult;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public File saveTo( File aFile, String aComment, char aDelimiter ) throws IOException {
		try (OutputStream theOutputStream = new BufferedOutputStream( new FileOutputStream( aFile ) )) {
			saveTo( theOutputStream, aComment, aDelimiter );
		}
		_propertiesFile = aFile;
		return aFile;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties seekFrom( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {

		List<String> theProbedFiles = new ArrayList<>();
		IOException cause = null;
		Properties theResult = null;

		// |--> Absolute path?
		File eFile = new File( aFilePath );
		theProbedFiles.add( eFile.getAbsolutePath() );
		if ( eFile.exists() && eFile.isFile() && eFile.canRead() ) {
			theResult = loadFrom( eFile );
			return theResult;
		}
		// Absolute path? <--|
		else {
			List<File> theConfigDirs = RuntimeUtility.toConfigDirs( eFile );
			// Try files |-->
			for ( File eDir : theConfigDirs ) {
				eFile = new File( eDir, aFilePath );
				theProbedFiles.add( eFile.getAbsolutePath() );
				if ( eFile.exists() && eFile.isFile() && eFile.canRead() ) {
					theResult = loadFrom( eFile );
					_propertiesFile = eFile;
					return theResult;
				}
			}
			// Try files <--|

			// Try JAR |-->
			if ( aResourceClass != null ) {
				URL theResource = aResourceClass.getResource( aFilePath );
				try {
					InputStream theInputStream = aResourceClass.getResourceAsStream( aFilePath );

					if ( theInputStream == null && !aFilePath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
						theInputStream = aResourceClass.getResourceAsStream( Delimiter.PATH.getChar() + aFilePath );
					}
					if ( theInputStream != null ) {
						theResult = loadFrom( theInputStream );
						if ( theResource != null ) {
							try {
								_propertiesFile = new File( theResource.toURI() );
							}
							catch ( URISyntaxException e ) {
								/* at least we were able to load! */
							}
						}
						return theResult;
					}
				}
				catch ( IOException e ) {
					cause = e;
				}
				theProbedFiles.add( theResource != null ? theResource.toExternalForm() : Scheme.JAR.getMarker() + aFilePath + "@" + aResourceClass.getName() );
			}
			// Try JAR <--|

			// Nothing found |-->
			StringBuilder theProbedLocations = new StringBuilder();
			Iterator<String> e = theProbedFiles.iterator();
			while ( e.hasNext() ) {
				theProbedLocations.append( e.next() );
				if ( e.hasNext() ) theProbedLocations.append( ", " );
			}
			throw new IOException( "Cannot find a properties file <" + aFilePath + "> at any of the supported locations: " + theProbedLocations.toString(), cause );
			// Nothing found <--|
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		try {
			saveTo( _propertiesFile );
		}
		catch ( IOException e ) {
			throw new OpenException( e.getMessage(), e );
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * In case the resource has not been loaded from a writable {@link File},
	 * then calling this method will return false.
	 */
	@Override
	public boolean isFlushable() {
		return _propertiesFile != null && _propertiesFile.canWrite();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties reload() throws IOException, IllegalStateException, ParseException {
		return reload( ReloadMode.KEEP_ORPHANS );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties reload( ReloadMode aReloadMode ) throws IOException, IllegalStateException, ParseException {
		if ( _propertiesFile == null ) {
			throw new IllegalArgumentException( "There is no resource (supporting reloading, such as an input stream) attached to these properties." );
		}
		Properties theResult = loadFrom( _propertiesFile );
		if ( aReloadMode == ReloadMode.ORPHAN_REMOVAL ) {
			Set<String> theKeys = new HashSet<>( keySet() );
			for ( String eKey : theKeys ) {
				if ( !theResult.containsKey( eKey ) ) {
					remove( eKey );
				}
			}
		}
		return theResult;
	}

	// /////////////////////////////////////////////////////////////////////////
	// SUB-TYPED:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}