// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import org.refcodes.data.Delimiter;
import org.refcodes.structure.PathMap;
import org.refcodes.structure.Relation;

/**
 * The {@link ProfileProperties} extend the {@link Properties} with
 * Runtime-Profiles support. Properties belonging to provided Runtime-Profiles
 * overwrite the default properties. Use {@link #toRuntimeProfile()} to generate
 * accordingly processed Runtime-Profiles with Runtime-Profiles defined below
 * the {@link #getRuntimeProfilesPath()}, which by default is
 * "/runtime/profiles".
 * 
 * As more than one profile can be evaluated (a preceding profile overwriting
 * succeeding profile), the {@link #getRuntimeProfilesPath()} points to an array
 * with the Runtime-Profiles denoted as follows (assuming "/runtime/profiles"
 * containing your active Runtime-Profiles):
 * 
 * <ul>
 * <li>runtime/profiles/0=local</li>
 * <li>runtime/profiles/1=oauth</li>
 * <li>runtime/profiles/2=development</li>
 * </ul>
 * 
 * ({@link #getRuntimeProfilesPath()} defaults to "runtime/profiles" represented
 * by the path defined by {@link PropertiesPath#RUNTIME_PROFILES_PATH})
 * 
 * In case a value is found at the path "runtime/profiles", this is parsed as a
 * comma-separated list and converted to an array. The above configuration
 * therewith may also be written as follows:
 * 
 * <code>runtime/profiles=local,oauth,development</code>
 * 
 * This array beats the above kind of declaration so that the active
 * Runtime-Profiles can easily be overwritten by {@link ProfileProperties}
 * supporting Java System-Properties (passed to the JVM via "-Dx=y") e.g. the
 * <code>RuntimePropertiesComposite</code> found in the
 * <code>refcodes-runtime</code> artifact.
 * 
 * Below find an example on properties with settings for the above
 * Runtime-Profiles:
 * 
 * <ul>
 * <li>db_url=someStagingDbUrl</li>
 * <li>db_user=admin</li>
 * <li>security=false</li>
 * <li>oauth/security=true</li>
 * <li>local/db_user=devops</li>
 * <li>local/db_url=someLocalDbUrl</li>
 * <li>development/db_user=admin</li>
 * <li>development/db_url=someDevelopmentDbUrl</li>
 * <li>test/db_user=test</li>
 * <li>test/db_url=someTestDbUrl</li>
 * </ul>
 * 
 * In the above example, the default settings would be:
 * 
 * <ul>
 * <li>db_url=someStagingDbUrl</li>
 * <li>db_user=admin</li>
 * </ul>
 * 
 * The default settings are replaced / enriched by the settings found in the
 * "local" profile, any settings not found for the "local" profile are
 * overwritten / enriched by the "oauth" profile. Any settings not found in the
 * "oauth" profile (and not found in the "local" profile") are overwritten /
 * enriched by the "development" profile.
 * 
 * The properties created from the active Runtime-Profiles specified below the
 * path "runtime/profiles" via {@link #toRuntimeProfile()} result in the below
 * default properties:
 * 
 * <ul>
 * <li>/db_url:= someLocalDbUrl</li>
 * <li>/db_user:= devops</li>
 * <li>/security:= true</li>
 * </ul>
 */
public interface ProfileProperties extends Properties {

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns the path which points to your active Runtime-Profiles. The method
	 * {@link #getRuntimeProfiles()} evaluates the path and creates a list of
	 * active Runtime-Profiles. By default, the path points to the path
	 * represented by {@link PropertiesPath#RUNTIME_PROFILES_PATH}.
	 * 
	 * @return The path pointing to your active Runtime-Profiles, defaults to
	 *         "runtime/profiles".
	 */
	default String getRuntimeProfilesPath() {
		return PropertiesPath.RUNTIME_PROFILES_PATH.getPath();
	}

	/**
	 * The {@link #getRuntimeProfilesPath()} points to an array with the
	 * Runtime-Profiles denoted as follows (pointing to your active
	 * Runtime-Profiles):
	 * 
	 * <ul>
	 * <li>runtime/profiles/0=local</li>
	 * <li>runtime/profiles/1=oauth</li>
	 * <li>runtime/profiles/2=development</li>
	 * </ul>
	 * 
	 * ({@link #getRuntimeProfilesPath()} defaults to "runtime/profiles"
	 * represented by the path defined by
	 * {@link PropertiesPath#RUNTIME_PROFILES_PATH})
	 * 
	 * In case a value is found at the path "runtime/profiles", this is parsed
	 * as a comma-separated list and converted to an array. The above
	 * configuration therewith may also be written as follows:
	 * 
	 * <code>runtime/profiles=local,oauth,development</code>
	 * 
	 * This array beats the above kind of declaration so that the active
	 * Runtime-Profiles can easily be overwritten by {@link ProfileProperties}
	 * supporting Java System-Properties (passed to the JVM via "-Dx=y") e.g.
	 * the <code>RuntimePropertiesComposite</code> found in the
	 * <code>refcodes-runtime</code> artifact.
	 * 
	 * @return Your active Runtime-Profiles defined at the
	 *         {@link #getRuntimeProfilesPath()}.
	 */
	default String[] getRuntimeProfiles() {
		String theRecord = get( getRuntimeProfilesPath() );
		if ( theRecord != null ) {
			String[] theProfiles = theRecord.split( Delimiter.LIST.getChar() + "" );
			if ( theProfiles != null && theProfiles.length != 0 ) {
				return theProfiles;
			}
		}
		return getArray( getRuntimeProfilesPath() );
	}

	/**
	 * Evaluates the active Runtime-Profiles as of {@link #getRuntimeProfiles()}
	 * and creates the according {@link Properties}.
	 * 
	 * @return The {@link Properties} as of the active Runtime-Profiles.
	 */
	default Properties toRuntimeProfile() {
		return toRuntimeProfile( getRuntimeProfiles() );
	}

	/**
	 * Evaluates the provided Runtime-Profiles and creates the according
	 * {@link Properties}.
	 *
	 * @param aProfiles the profiles
	 * @return The {@link Properties} as of the provided Runtime-Profiles.
	 */
	default Properties toRuntimeProfile( String... aProfiles ) {
		PropertiesBuilder theProperties = new PropertiesBuilderImpl( this );
		String eKey;
		if ( aProfiles != null ) {
			for ( int i = aProfiles.length - 1; i >= 0; i-- ) {
				eKey = aProfiles[i];
				if ( eKey != null && eKey.length() != 0 ) {
					// theProperties.removeFrom( eKey );
					PathMap<String> theRetrieved = retrieveFrom( eKey );
					theProperties.insert( theRetrieved );
				}
			}
		}
		return new PropertiesImpl( (Properties) theProperties );
	}

	// /////////////////////////////////////////////////////////////////////////
	// MUTATOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Interface MutableProfileProperties.
	 */
	public interface MutableProfileProperties extends ProfileProperties, MutableProperties {}

	// /////////////////////////////////////////////////////////////////////////
	// BUILDER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Interface ProfilePropertiesBuilder.
	 */
	public interface ProfilePropertiesBuilder extends MutableProfileProperties, PropertiesBuilder {

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// SUB-TYPED METHODS:
		// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPut( String aKey, String aValue ) {
			put( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPut( Relation<String, String> aProperty ) {
			put( aProperty );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutInteger( String aKey, Integer aValue ) {
			putInteger( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutShort( String aKey, Short aValue ) {
			putShort( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutByte( String aKey, Byte aValue ) {
			putByte( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutDouble( String aKey, Double aValue ) {
			putDouble( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutFloat( String aKey, Float aValue ) {
			putFloat( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutBoolean( String aKey, Boolean aValue ) {
			putBoolean( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withPutLong( String aKey, Long aValue ) {
			putLong( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withInsert( Object aObj ) {
			insert( aObj );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withInsertFrom( Object aFrom, String aFromPath ) {
			insertFrom( aFrom, aFromPath );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withInsertTo( String aToPath, Object aFrom ) {
			insertTo( aToPath, aFrom );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withInsert( String aToPath, Object aFrom, String aFromPath ) {
			insert( aToPath, aFrom, aFromPath );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default ProfilePropertiesBuilder withRemoveFrom( String aPath ) {
			removeAll( aPath );
			return this;
		}
	}
}
