// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.util.HashMap;
import java.util.Map;

import org.refcodes.data.Delimiter;
import org.refcodes.data.Text;
import org.refcodes.structure.CanonicalMap;
import org.refcodes.structure.Dictionary;
import org.refcodes.structure.Property;
import org.refcodes.structure.Relation;

/**
 * The {@link Properties} are a specialization of the {@link Dictionary}.
 * Properties represent a collection of {@link String} key and {@link String}
 * value pairs as properties usually occur in pure text form, to be converted to
 * the required data types. For this reason, the {@link Properties} interface
 * provides additional conversion methods.
 */
public interface Properties extends CanonicalMap {

	public static final String DEFAULT_COMMENT = "Generated by <" + Text.REFCODES_ORG + "> (http://www.refcodes.org)";

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns the default path delimiter as of {@link Delimiter#PATH}.
	 * 
	 * {@inheritDoc}
	 */
	@Override
	default char getDelimiter() {
		return Delimiter.PATH.getChar();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Class<String> getType() {
		return String.class;
	}

	/**
	 * This method creates a {@link java.util.Properties} instance from this
	 * {@link Properties} instance's elements (key/value-pairs) as of
	 * interoperability reasons.
	 * 
	 * @return A {@link java.util.Properties} object from the herein contained
	 *         key/value-pairs .
	 */
	default java.util.Properties toProperties() {
		java.util.Properties theProperties = new java.util.Properties();
		for ( String eKey : keySet() ) {
			theProperties.put( toPropertyPath( eKey ), get( eKey ) );
		}
		return theProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default boolean containsValue( Object value ) {
		return values().contains( value );
	}

	/**
	 * This method creates a {@link Map} instance from this {@link Properties}
	 * instance's elements (key/value-pairs) as of interoperability reasons.
	 * 
	 * @return A {@link Map} object from the herein contained key/value-pairs .
	 */
	@Override
	default Map<String, String> toMap() {
		Map<String, String> theProperties = new HashMap<>();
		for ( String eKey : keySet() ) {
			theProperties.put( eKey, get( eKey ) );
		}
		return theProperties;
	}

	/**
	 * Converts the properties for saving; that when saved, them properties will
	 * not start with a delimiter, making them (when there are no sub-paths for
	 * the properties) look just like normal well known properties, enabling
	 * interchangeability with other systems reading the properties.
	 * 
	 * @param aPath The path to be normalized.
	 * 
	 * @return The normalized path; afterwards it does not start with a
	 *         delimiter any more.
	 */
	default String toPropertyPath( String aPath ) {
		if ( aPath != null ) {
			while ( aPath.startsWith( getRootPath() ) ) {
				aPath = aPath.substring( 1 );
			}
			while ( aPath.endsWith( getRootPath() ) ) {
				aPath = aPath.substring( 0, aPath.length() - 1 );
			}
			return aPath;
		}
		return null;
	}

	// /////////////////////////////////////////////////////////////////////////
	// SUB-TYPED:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	Properties retrieveFrom( String aFromPath );

	/**
	 * {@inheritDoc}
	 */
	@Override
	Properties retrieveTo( String aToPath );

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties childrenOf( String aParentPath );

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Properties retrieve( String aFromPath, String aToPath ) {
		Properties theProperties = retrieveFrom( aFromPath );
		return theProperties.retrieveTo( aToPath );
	}

	// /////////////////////////////////////////////////////////////////////////
	// MUTATOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The interface {@link MutableProperties} defines "dirty" methods allowing
	 * to modify ("mutate") the properties.
	 */
	public interface MutableProperties extends Properties, MutableCanonicalMap {

		/**
		 * {@inheritDoc}
		 */
		@Override
		default String put( Relation<String, String> aRelation ) {
			return put( aRelation.getKey(), aRelation.getValue() );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default String delete( String aKey ) {
			return remove( aKey );
		}

		/**
		 * Removes a property identified by the key of the provided
		 * {@link Property} (the value of the {@link Property} is ignored).
		 * 
		 * @param aProperty The {@link Property} which's key is to be removed.
		 * 
		 * @return The removed value of the key or null if there was not such a
		 *         key.
		 */
		default String delete( Property aProperty ) {
			if ( aProperty != null ) { return remove( aProperty.getKey() ); }
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default boolean containsValue( Object value ) {
			return values().contains( value );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default void putAll( Map<? extends String, ? extends String> aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, aProperties.get( (String) eKey ) );
			}
		}

		/**
		 * This method inserts all elements (key/value-pairs) found in the
		 * provided {@link Properties} instances of interoperability reasons.
		 * 
		 * @param aProperties A {@link Properties} containing the key/value
		 *        pairs to be inserted.
		 */
		default void putAll( Properties aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, aProperties.get( (String) eKey ) );
			}
		}

		/**
		 * This method inserts all elements (key/value-pairs) found in the
		 * provided {@link java.util.Properties} instances of interoperability
		 * reasons.
		 * 
		 * @param aProperties A {@link java.util.Properties} containing the
		 *        key/value-pairs to be inserted.
		 */
		@Override
		default void putAll( java.util.Properties aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, (String) aProperties.get( eKey ) );
			}
		}

		// /////////////////////////////////////////////////////////////////////
		// SUB-TYPED:
		// /////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////
		// SEMANTIC:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insert(Object)}.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 */
		default void insert( Properties aFrom ) {
			insert( (Object) aFrom );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insertFrom(Object, String)}.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @param aFromPath The path from where to start adding elements of the
		 *        provided object.
		 */
		default void insertFrom( Properties aFrom, String aFromPath ) {
			insertFrom( (Object) aFrom, aFromPath );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insertTo(String, Object)}.
		 * 
		 * @param aToPath The sub-path where to insert the object's introspected
		 *        values to.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 */
		default void insertTo( String aToPath, Properties aFrom ) {
			insertTo( aToPath, (Object) aFrom );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insert(String, Object, String)}.
		 * 
		 * @param aToPath The sub-path where to insert the object's introspected
		 *        values to.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @param aFromPath The path from where to start adding elements of the
		 *        provided object.
		 */
		default void insert( String aToPath, Properties aFrom, String aFromPath ) {
			insert( aToPath, (Object) aFrom, aFromPath );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// BUILDER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The interface {@link PropertiesBuilder} defines builder functionality on
	 * top of the properties .
	 */
	public interface PropertiesBuilder extends CanonicalMapBuilder, MutableProperties {

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////
		// SUB-TYPED:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPut( String aKey, String aValue ) {
			put( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPut( Relation<String, String> aProperty ) {
			put( aProperty );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutInteger( String aKey, Integer aValue ) {
			putInteger( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutShort( String aKey, Short aValue ) {
			putShort( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutByte( String aKey, Byte aValue ) {
			putByte( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutDouble( String aKey, Double aValue ) {
			putDouble( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutFloat( String aKey, Float aValue ) {
			putFloat( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutBoolean( String aKey, Boolean aValue ) {
			putBoolean( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withPutLong( String aKey, Long aValue ) {
			putLong( aKey, aValue );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withInsert( Object aObj ) {
			insert( aObj );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withInsertFrom( Object aFrom, String aFromPath ) {
			insertFrom( aFrom, aFromPath );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withInsertTo( String aToPath, Object aFrom ) {
			insertTo( aToPath, aFrom );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withInsert( String aToPath, Object aFrom, String aFromPath ) {
			insert( aToPath, aFrom, aFromPath );
			return this;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		default PropertiesBuilder withRemoveFrom( String aPath ) {
			removeAll( aPath );
			return this;
		}

		// /////////////////////////////////////////////////////////////////////
		// SEMANTIC:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insert(Object)}.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @return This instance as of the Builder-Pattern to apply succeeding
		 *         operations.
		 * 
		 * @see #insert(Object)
		 */
		default PropertiesBuilder withInsert( Properties aFrom ) {
			return withInsert( (Object) aFrom );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insertFrom(Object, String)}.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @param aFromPath The path from where to start adding elements of the
		 *        provided object.
		 * 
		 * @return This instance as of the Builder-Pattern to apply succeeding
		 *         operations.
		 * 
		 * @see #insertFrom(Object, String)
		 */
		default PropertiesBuilder withInsertFrom( Properties aFrom, String aFromPath ) {
			return withInsertFrom( (Object) aFrom, aFromPath );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insertTo(String, Object)}.
		 * 
		 * @param aToPath The sub-path where to insert the object's introspected
		 *        values to.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @return This instance as of the Builder-Pattern to apply succeeding
		 *         operations.
		 * 
		 * @see #insertTo(String, Object)
		 */
		default PropertiesBuilder withInsertTo( String aToPath, Properties aFrom ) {
			return withInsertTo( aToPath, (Object) aFrom );
		}

		/**
		 * Method to semantically emphasize that we support our own types.
		 * Actually delegates to {@link #insert(String, Object, String)}.
		 * 
		 * @param aToPath The sub-path where to insert the object's introspected
		 *        values to.
		 * 
		 * @param aFrom The {@link Properties} which is to be inspected with the
		 *        therein contained values being added with their according
		 *        determined paths.
		 * 
		 * @param aFromPath The path from where to start adding elements of the
		 *        provided object.
		 * 
		 * @return This instance as of the Builder-Pattern to apply succeeding
		 *         operations.
		 * 
		 * @see #insert(String, Object, String)
		 */
		default PropertiesBuilder withInsert( String aToPath, Properties aFrom, String aFromPath ) {
			return withInsert( aToPath, (Object) aFrom, aFromPath );
		}
	}
}
