// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.refcodes.exception.IntegrityException;

/**
 * Retrieve properties from various different properties sources
 * ({@link Properties} instances) by querying all the herein contained
 * {@link Properties} instances in the order of them being added. Queried
 * properties of the first {@link Properties} instance containing them are
 * returned. {@link Properties} instances before have a higher precedence than
 * the instances added next.
 */
public class PropertiesPrecedenceComposite implements PropertiesPrecedence {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected List<Properties> _properties;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a {@link PropertiesPrecedence} composite querying the provided
	 * {@link Properties} in the given order. Queried properties of the first
	 * {@link Properties} instance containing them are returned.
	 * {@link Properties} before have a higher precedence over
	 * {@link Properties} provided next.
	 * 
	 * @param aProperties The {@link Properties} to be queried in the provided
	 *        order.
	 */
	public PropertiesPrecedenceComposite( Properties... aProperties ) {
		_properties = new ArrayList<>( Arrays.asList( aProperties ) );
	}

	/**
	 * Creates a {@link PropertiesPrecedence} composite querying the provided
	 * {@link Properties} in the given order. Queried properties of the first
	 * {@link Properties} instance containing them are returned.
	 * {@link Properties} before have a higher precedence over
	 * {@link Properties} provided next.
	 * 
	 * @param aProperties The {@link Properties} to be queried in the provided
	 *        order.
	 */
	public PropertiesPrecedenceComposite( List<Properties> aProperties ) {
		_properties = aProperties;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object toDataStructure( String aFromPath ) {
		PropertiesBuilder theProperties = new PropertiesBuilderImpl();
		for ( int i = _properties.size() - 1; i >= 0; i-- ) {
			theProperties.insert( _properties.get( i ).retrieveFrom( aFromPath ) );
		}
		return theProperties.toDataStructure();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean containsKey( Object aKey ) {
		for ( int i = 0; i < _properties.size(); i++ ) {
			if ( _properties.get( i ).containsKey( aKey ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String get( Object aKey ) {
		Properties eProperties;
		for ( int i = 0; i < _properties.size(); i++ ) {
			eProperties = _properties.get( i );
			if ( eProperties.containsKey( aKey ) ) {
				return eProperties.get( aKey );
			}
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<String> keySet() {
		Set<String> theSet = new LinkedHashSet<>();
		for ( int i = 0; i < _properties.size(); i++ ) {
			theSet.addAll( _properties.get( i ).keySet() );
		}
		return theSet;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<String> values() {
		Collection<String> theCollection = new ArrayList<>();
		for ( int i = 0; i < _properties.size(); i++ ) {
			theCollection.addAll( _properties.get( i ).values() );
		}
		return theCollection;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int size() {
		int size = 0;
		for ( int i = 0; i < _properties.size(); i++ ) {
			size += _properties.get( i ).size();
		}
		return size;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEmpty() {
		for ( int i = 0; i < _properties.size(); i++ ) {
			if ( !_properties.get( i ).isEmpty() ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public char getDelimiter() {
		Character theDelimiter = null;
		char eDelimiter;
		for ( int i = 0; i < _properties.size(); i++ ) {
			eDelimiter = _properties.get( i ).getDelimiter();
			if ( theDelimiter != null && !theDelimiter.equals( eDelimiter ) ) {
				throw new IntegrityException( "The encapsulated properties define different ambigous delimiter <" + theDelimiter + "> and <" + eDelimiter + "> defintions." );
			}
			theDelimiter = eDelimiter;
		}
		return theDelimiter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties retrieveFrom( String aFromPath ) {
		PropertiesBuilder theBuilder = new PropertiesBuilderImpl();
		for ( int i = _properties.size() - 1; i >= 0; i-- ) {
			theBuilder.insert( _properties.get( i ).retrieveFrom( aFromPath ) );
		}
		return theBuilder;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties retrieveTo( String aToPath ) {
		PropertiesBuilder theBuilder = new PropertiesBuilderImpl();
		for ( int i = _properties.size() - 1; i >= 0; i-- ) {
			theBuilder.insert( _properties.get( i ).retrieveTo( aToPath ) );
		}
		return theBuilder;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties childrenOf( String aParentPath ) {
		PropertiesBuilder theBuilder = new PropertiesBuilderImpl();
		for ( int i = _properties.size() - 1; i >= 0; i-- ) {
			theBuilder.insert( _properties.get( i ).childrenOf( aParentPath ) );
		}
		return theBuilder;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
