// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.text.ParseException;

import org.refcodes.runtime.RuntimeUtility;

/**
 * This mixin provides builder additions (as of the builder pattern for chained
 * configuring method calls) for "dynamic" {@link ResourceProperties}: As
 * {@link ResourceProperties} are immutable from an interface's point of view,
 * there are no mutating methods provided. Sub-types of the
 * {@link ResourceProperties} might need to load the properties after
 * instantiation. Such types might implement this interface, providing means to
 * load from resources after instantiation.
 */
public interface ResourcePropertiesMixin<B extends ResourcePropertiesMixin<B>> {

	/**
	 * Loads a properties file from the file directly or (if not found) from
	 * first folder containing such a file as of the specification for the
	 * method {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The file of the properties to load.
	 * 
	 * @return This instance as of the builder pattern for chained method calls.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	B withFile( File aFile ) throws IOException, ParseException;

	/**
	 * Reads the properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @return This instance as of the builder pattern for chained method calls.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	B withInputStream( InputStream aInputStream ) throws IOException, ParseException;

	/**
	 * Loads a properties file from the file directly or (if not found) from
	 * first folder containing such a file as of the specification for the
	 * method {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The file of the properties file to load.
	 * 
	 * @return This instance as of the builder pattern for chained method calls.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	B withFilePath( String aFilePath ) throws IOException, ParseException;

	/**
	 * Loads the properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @return This instance as of the builder pattern for chained method calls.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	B withUrl( URL aUrl ) throws IOException, ParseException;

}
