// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import org.refcodes.structure.KeyNotFoundRuntimeException;
import org.refcodes.structure.Relation;

/**
 * The {@link StrictProperties} extends the {@link Properties} with all the
 * getters to throw a {@link KeyNotFoundRuntimeException} instead of returning
 * null in case the key was not found.
 */
public interface StrictProperties extends Properties {

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Character getCharacter( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getCharacter( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Integer getInteger( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getInteger( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Long getLong( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getLong( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Short getShort( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getShort( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Byte getByte( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getByte( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Double getDouble( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getDouble( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Float getFloat( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getFloat( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Boolean getBoolean( String aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey, "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getBoolean( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default String get( Enum<?> aKey ) throws KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.get( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Character getCharacter( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getCharacter( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Integer getInteger( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getInteger( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Short getShort( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getShort( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Byte getByte( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getByte( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Double getDouble( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getDouble( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Float getFloat( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getFloat( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Boolean getBoolean( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getBoolean( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default Long getLong( Enum<?> aKey ) throws NumberFormatException, KeyNotFoundRuntimeException {
		if ( !containsKey( aKey ) ) {
			throw new KeyNotFoundRuntimeException( aKey.toString(), "There is no such element with key <" + aKey + "> found in this instance!" );
		}
		return Properties.super.getLong( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	String get( Object aKey ) throws KeyNotFoundRuntimeException;

	// /////////////////////////////////////////////////////////////////////////
	// MUTATOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Interface MutableStrictProperties.
	 */
	public interface MutableStrictProperties extends StrictProperties, MutableProperties {}

	// /////////////////////////////////////////////////////////////////////////
	// BUILDER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Interface StrictPropertiesBuilder.
	 */
	public interface StrictPropertiesBuilder extends MutableStrictProperties, PropertiesBuilder {

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// SUB-TYPED METHODS:
		// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPut( String aKey, String aValue ) {
			put( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPut( Relation<String, String> aProperty ) {
			put( aProperty );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutInteger( String aKey, Integer aValue ) {
			putInteger( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutShort( String aKey, Short aValue ) {
			putShort( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutByte( String aKey, Byte aValue ) {
			putByte( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutDouble( String aKey, Double aValue ) {
			putDouble( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutFloat( String aKey, Float aValue ) {
			putFloat( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutBoolean( String aKey, Boolean aValue ) {
			putBoolean( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withPutLong( String aKey, Long aValue ) {
			putLong( aKey, aValue );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withInsert( Object aObj ) {
			insert( aObj );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withInsertFrom( Object aFrom, String aFromPath ) {
			insertFrom( aFrom, aFromPath );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withInsertTo( String aToPath, Object aFrom ) {
			insertTo( aToPath, aFrom );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withInsert( String aToPath, Object aFrom, String aFromPath ) {
			insert( aToPath, aFrom, aFromPath );
			return this;
		}

	/**
	 * {@inheritDoc}
	 */
	@Override
		default StrictPropertiesBuilder withRemoveFrom( String aPath ) {
			removeAll( aPath );
			return this;
		}
	}
}
