// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.text.ParseException;
import java.util.Collection;
import java.util.Set;

import org.refcodes.runtime.RuntimeUtility;

/**
 * Implementation of the {@link ResourceProperties} interface with support of so
 * called "TOML properties".
 * 
 * For TOML properties, see "https://en.wikipedia.org/wiki/TOML"
 */
public class TomlProperties implements ResourceProperties {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected ResourcePropertiesBuilder _properties;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Loads the TOML Properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( File aFile ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFile );
	}

	/**
	 * Loads the TOML Properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( File aFile, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the TOML Properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( File aFile, boolean isSeekFile ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFile, isSeekFile );
	}

	/**
	 * Loads or seeks the TOML Properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( File aFile, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFile, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( String aFilePath ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFilePath );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFilePath, aDelimiters );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aResourceClass, aFilePath );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFilePath, isSeekFile );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( Class<?> aResourceClass, String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aResourceClass, aFilePath, isSeekFile );
	}

	/**
	 * Loads the TOML Properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( Class<?> aResourceClass, String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aResourceClass, aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML Properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( URL aUrl ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aUrl );
	}

	/**
	 * Loads the TOML Properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aUrl, aDelimiters );
	}

	/**
	 * Reads the TOML Properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( InputStream aInputStream ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aInputStream );
	}

	/**
	 * Reads the TOML Properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlProperties( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		_properties = new TomlPropertiesBuilder( aInputStream, aDelimiters );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean containsKey( Object aKey ) {
		return _properties.containsKey( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String get( Object aKey ) {
		return _properties.get( aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<String> keySet() {
		return _properties.keySet();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<String> values() {
		return _properties.values();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int size() {
		return _properties.size();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEmpty() {
		return _properties.isEmpty();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties retrieveFrom( String aFromPath ) {
		return _properties.retrieveFrom( aFromPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties retrieveTo( String aToPath ) {
		return _properties.retrieveTo( aToPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties childrenOf( String aParentPath ) {
		return _properties.childrenOf( aParentPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public char getDelimiter() {
		return _properties.getDelimiter();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<String> getType() {
		return _properties.getType();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object toDataStructure( String aFromPath ) {
		return _properties.toDataStructure( aFromPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties reload() throws IOException, IllegalStateException, ParseException {
		return _properties.reload();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties reload( ReloadMode aReloadMode ) throws IOException, IllegalStateException, ParseException {
		return _properties.reload( aReloadMode );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
