// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.refcodes.configuration.ResourceProperties.ResourcePropertiesBuilder;
import org.refcodes.data.Delimiter;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.structure.Property;
import org.refcodes.structure.PropertyImpl;

/**
 * Implementation of the {@link ResourcePropertiesBuilder} interface with
 * support of so called "TOML properties".
 * 
 * For TOML properties, see "https://en.wikipedia.org/wiki/TOML"
 */
public class TomlPropertiesBuilder extends AbstractResourcePropertiesBuilder implements ResourcePropertiesBuilder {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	public static final char SECTION_BEGIN = '[';
	public static final char SECTION_END = ']';
	public static final char COMMENT = '#';

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Create an empty {@link TomlPropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations.
	 */
	public TomlPropertiesBuilder() {
		super();
	}

	/**
	 * Create a {@link TomlPropertiesBuilder} instance containing the
	 * elements as of {@link MutablePathMap#insert(Object)} using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations:
	 * 
	 * "Inspects the given object and adds all elements found in the given
	 * object. Elements of type {@link Map}, {@link Collection} and arrays are
	 * identified and handled as of their type: The path for each value in a
	 * {@link Map} is appended with its according key. The path for each value
	 * in a {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aObj The object from which the elements are to be added.
	 */
	public TomlPropertiesBuilder( Object aObj ) {
		super( aObj );
	}

	/**
	 * Create a {@link TomlPropertiesBuilder} instance containing the
	 * elements of the provided {@link Properties} instance using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public TomlPropertiesBuilder( Properties aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link TomlPropertiesBuilder} instance containing the
	 * elements of the provided {@link PropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public TomlPropertiesBuilder( PropertiesBuilder aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link TomlPropertiesBuilder} instance containing the
	 * elements of the provided {@link Map} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public TomlPropertiesBuilder( Map<?, ?> aProperties ) {
		super( aProperties );
	}

	/**
	 * Loads the TOML properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( File aFile ) throws IOException, ParseException {
		super( aFile );
	}

	/**
	 * Loads the TOML properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( File aFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the TOML properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( File aFile, boolean isSeekFile ) throws IOException, ParseException {
		super( aFile, isSeekFile );
	}

	/**
	 * Loads or seeks the TOML properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( File aFile, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFile, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( String aFilePath ) throws IOException, ParseException {
		super( aFilePath );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aFilePath, aDelimiters );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		super( aResourceClass, aFilePath );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		super( aFilePath, isSeekFile );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, isSeekFile );
	}

	/**
	 * Loads the TOML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed). *
	 * 
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the TOML properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( URL aUrl ) throws IOException, ParseException {
		super( aUrl );
	}

	/**
	 * Loads the TOML properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		super( aUrl, aDelimiters );
	}

	/**
	 * Reads the TOML properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( InputStream aInputStream ) throws IOException, ParseException {
		super( aInputStream );
	}

	/**
	 * Reads the TOML properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public TomlPropertiesBuilder( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		super( aInputStream, aDelimiters );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties loadFrom( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		BufferedReader theReader = new BufferedReader( new InputStreamReader( aInputStream ) );
		String eLine, eTruncated;
		String eSection = "";
		Property eProperty;
		int eLineNum = 0;
		PropertiesBuilder theResult = new PropertiesBuilderImpl();
		while ( theReader.ready() ) {
			eLineNum++;
			eLine = theReader.readLine();
			eTruncated = eLine.replaceAll( "^\\s+", "" );
			if ( !eTruncated.startsWith( "" + COMMENT ) ) {
				eTruncated = eLine.trim();
				if ( eTruncated.length() > 0 ) {
					if ( eTruncated.startsWith( "" + SECTION_BEGIN ) && eTruncated.endsWith( "" + SECTION_END ) ) {
						eSection = getDelimiter() + eTruncated.substring( 1, eTruncated.length() - 1 );
					}
					else {

						if ( !eLine.contains( Delimiter.PROPERTY.getChar() + "" ) ) {
							throw new ParseException( "Expected a \"" + Delimiter.PROPERTY.getChar() + "\" at line < " + eLineNum + ">, line cannot be parsed as property.", eLineNum );
						}
						eProperty = new PropertyImpl( eLine );

						if ( eProperty.getKey().contains( " " ) ) {
							throw new ParseException( "The key \"" + eProperty.getKey() + "\" contains a space \" \" at line < " + eLineNum + ">, a key must not contain spaces.", eLineNum );
						}
						eTruncated = eProperty.getKey().trim();
						eTruncated = fromExternalKey( eTruncated, aDelimiters );
						theResult.put( eSection + getDelimiter() + eTruncated, eProperty.getValue() );
					}
				}
			}
		}
		insert( theResult );
		return theResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveTo( OutputStream aOutputStream, String aComment, char aDelimiter ) throws IOException {
		List<String> theGlobals = new ArrayList<>( records() );
		Collections.sort( theGlobals );
		List<String> theSections = new ArrayList<>( directories() );
		Collections.sort( theSections );
		boolean hasWritten = false;
		try (BufferedWriter theWriter = new BufferedWriter( new OutputStreamWriter( aOutputStream ) )) {

			// Write the comment |-->
			if ( aComment != null && aComment.length() > 0 ) {
				String theLines[] = aComment.split( "\\r?\\n" );
				for ( String eLine : theLines ) {
					theWriter.write( COMMENT + " " + eLine );
					theWriter.newLine();
				}
				hasWritten = true;
			}
			// Write the comment <--|

			String eExternalKey;

			// Write the global properties |-->
			if ( theGlobals != null && !theGlobals.isEmpty() ) {
				if ( hasWritten ) {
					theWriter.newLine();
				}
				for ( String eKey : theGlobals ) {
					eExternalKey = toExternalKey( eKey, aDelimiter );
					theWriter.write( eExternalKey + Delimiter.PROPERTY.getChar() + get( eKey ) );
					theWriter.newLine();
				}
				hasWritten = true;
			}
			// Write the global properties <--|
			Properties eSecionProperties;
			List<String> eSectionKeys;
			for ( String eKey : theSections ) {

				if ( hasWritten ) {
					theWriter.newLine();
				}
				theWriter.write( SECTION_BEGIN + eKey + SECTION_END );
				theWriter.newLine();
				theWriter.newLine();
				eSecionProperties = retrieveFrom( eKey );
				eSectionKeys = new ArrayList<>( eSecionProperties.keySet() );
				Collections.sort( eSectionKeys );
				for ( String eSectionKey : eSectionKeys ) {
					eExternalKey = toExternalKey( eSectionKey, aDelimiter );
					eExternalKey = toPropertyPath( eExternalKey );
					theWriter.write( eExternalKey + Delimiter.PROPERTY.getChar() + eSecionProperties.get( eSectionKey ) );
					theWriter.newLine();
				}
				hasWritten = true;
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// SUB-TYPED:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////
}