// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.configuration;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.json.JSONObject;
import org.json.XML;
import org.refcodes.configuration.ResourceProperties.ResourcePropertiesBuilder;
import org.refcodes.data.Delimiter;
import org.refcodes.runtime.RuntimeUtility;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Implementation of the {@link ResourcePropertiesBuilder} interface with
 * support of so called "XML properties" (or just "properties").
 * 
 * For XML, see "https://en.wikipedia.org/wiki/XML".
 */
public class XmlPropertiesBuilder extends AbstractResourcePropertiesBuilder implements ResourcePropertiesBuilder {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	protected static final String ROOT_ELEMENT = "properties";

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	public static final String COMMENT_OPEN = "<!--";
	public static final String COMMENT_CLOSE = "-->";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Create an empty {@link XmlPropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations.
	 */
	public XmlPropertiesBuilder() {
		super();
	}

	/**
	 * Create a {@link XmlPropertiesBuilder} instance containing the
	 * elements as of {@link MutablePathMap#insert(Object)} using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations:
	 * 
	 * "Inspects the given object and adds all elements found in the given
	 * object. Elements of type {@link Map}, {@link Collection} and arrays are
	 * identified and handled as of their type: The path for each value in a
	 * {@link Map} is appended with its according key. The path for each value
	 * in a {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aObj The object from which the elements are to be added.
	 */
	public XmlPropertiesBuilder( Object aObj ) {
		super( aObj );
	}

	/**
	 * Create a {@link XmlPropertiesBuilder} instance containing the
	 * elements of the provided {@link Properties} instance using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public XmlPropertiesBuilder( Properties aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link XmlPropertiesBuilder} instance containing the
	 * elements of the provided {@link PropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public XmlPropertiesBuilder( PropertiesBuilder aProperties ) {
		super( aProperties );
	}

	/**
	 * Create a {@link XmlPropertiesBuilder} instance containing the
	 * elements of the provided {@link Map} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public XmlPropertiesBuilder( Map<?, ?> aProperties ) {
		super( aProperties );
	}

	/**
	 * Loads the XML properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( File aFile ) throws IOException, ParseException {
		super( aFile );
	}

	/**
	 * Loads the XML properties from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( File aFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the XML properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( File aFile, boolean isSeekFile ) throws IOException, ParseException {
		super( aFile, isSeekFile );
	}

	/**
	 * Loads or seeks the XML properties from the given {@link File}.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( File aFile, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFile, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( String aFilePath ) throws IOException, ParseException {
		super( aFilePath );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aFilePath, aDelimiters );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		super( aResourceClass, aFilePath );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		super( aFilePath, isSeekFile );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, isSeekFile );
	}

	/**
	 * Loads the XML properties from the given file's path.
	 * 
	 * In case you enable the "seek" argument and directly loading from the
	 * {@link File} does not succeed, then loads from the first folder
	 * containing such a file as of the specification for the method
	 * {@link RuntimeUtility#toConfigDirs()}.
	 * 
	 * Finally (if nothing else succeeds) the properties are loaded by the
	 * provided class's class loader which takes care of loading the properties
	 * (in case the file path is a relative path, also the absolute path with a
	 * prefixed path delimiter "/" is probed). *
	 * 
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * 
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @param isSeekFile When true, then the properties are loaded as of the
	 *        "seek" functionality described above.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( Class<?> aResourceClass, String aFilePath, boolean isSeekFile, char... aDelimiters ) throws IOException, ParseException {
		super( aResourceClass, aFilePath, isSeekFile, aDelimiters );
	}

	/**
	 * Loads the XML properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( URL aUrl ) throws IOException, ParseException {
		super( aUrl );
	}

	/**
	 * Loads the XML properties from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		super( aUrl, aDelimiters );
	}

	/**
	 * Reads the XML properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( InputStream aInputStream ) throws IOException, ParseException {
		super( aInputStream );
	}

	/**
	 * Reads the XML properties from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 *
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public XmlPropertiesBuilder( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		super( aInputStream, aDelimiters );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Properties loadFrom( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		ObjectMapper theMapper = new ObjectMapper();
		// theMapper.enable( SerializationFeature.WRITE_ENUMS_USING_TO_STRING );
		// theMapper.enable( DeserializationFeature.READ_ENUMS_USING_TO_STRING
		// );

		// As of XML "array" support |-->
		JSONObject theJsonObj = XML.toJSONObject( toString( aInputStream ) );
		String theJson = theJsonObj.toString( 4 );
		// As of XML "array" support <--|
		try {
			Map<String, String> theProperties = theMapper.readValue( theJson, new TypeReference<Map<?, ?>>() {} );
			Properties theResult = new PropertiesImpl( theProperties );
			// Auto-detect whether we have a root element |-->
			Set<String> theDirs = theResult.directories();
			if ( theDirs.size() == 1 ) {
				theResult = theResult.retrieveFrom( theDirs.iterator().next() );
			}
			// Auto-detect whether we have a root element <--|
			putAll( theResult );
			return theResult;
		}
		catch ( JsonMappingException | JsonParseException e ) {
			throw new ParseException( e.getMessage(), e.getLocation().getColumnNr() );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveTo( OutputStream aOutputStream, String aComment, char aDelimiter ) throws IOException {
		ObjectMapper theMapper = new ObjectMapper();
		// theMapper.enable( SerializationFeature.WRITE_ENUMS_USING_TO_STRING );
		// theMapper.enable( DeserializationFeature.READ_ENUMS_USING_TO_STRING
		// );

		// Auto-detect whether we need a root element |-->
		Object theDataStructure;
		if ( directories().size() > 1 ) {
			theDataStructure = retrieveTo( ROOT_ELEMENT ).toDataStructure();
		}
		else {
			theDataStructure = toDataStructure();
		}
		// Auto-detect whether we need a root element <--|
		String theJson = theMapper.writeValueAsString( theDataStructure );
		// As of XML "array" support |-->
		JSONObject theJsonObj = new JSONObject( theJson );
		String theXml = XML.toString( theJsonObj );
		// As of XML "array" support <--|
		// Pretty print |-->
		theXml = toPrettyPrint( theXml );

		// Pretty print <--|
		try (BufferedWriter theWriter = new BufferedWriter( new OutputStreamWriter( aOutputStream ) )) {
			// Write the comment |-->
			if ( aComment != null && aComment.length() > 0 ) {
				String theLines[] = aComment.split( "\\r?\\n" );
				for ( String eLine : theLines ) {
					theWriter.write( COMMENT_OPEN + " " + eLine + " " + COMMENT_CLOSE );
					theWriter.newLine();
				}
			}
			// Write the comment <--|
			theWriter.write( theXml );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// SUB-TYPED:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static String toString( InputStream aInputStream ) throws IOException {
		ByteArrayOutputStream theBuffer = new ByteArrayOutputStream();
		int eRead;
		byte[] theData = new byte[16384];
		while ( (eRead = aInputStream.read( theData, 0, theData.length )) != -1 ) {
			theBuffer.write( theData, 0, eRead );
		}
		theBuffer.flush();
		return theBuffer.toString();
	}

	private static String toPrettyPrint( String aXml ) {
		// TODO Implement pretty printing:
		return aXml;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

}