// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console;

import java.io.PrintStream;
import java.util.List;

import org.refcodes.component.Resetable;
import org.refcodes.mixin.DescriptionAccessor.DescriptionBuilder;
import org.refcodes.mixin.DescriptionAccessor.DescriptionMutator;
import org.refcodes.mixin.NameAccessor.NameBuilder;
import org.refcodes.mixin.NameAccessor.NameMutator;
import org.refcodes.mixin.TitleAccessor.TitleBuilder;
import org.refcodes.mixin.TitleAccessor.TitleMutator;
import org.refcodes.runtime.SystemUtility;
import org.refcodes.textual.Font;

/**
 * The {@link ArgsParser} provides means for parsing command line arguments and
 * constructing a command line utility's help output.
 *
 */
public interface ArgsParser extends TitleMutator, TitleBuilder<ArgsParser>, NameMutator, NameBuilder<ArgsParser>, DescriptionMutator, DescriptionBuilder<ArgsParser>, Resetable {

	/**
	 * Evaluates the provided command line arguments and determines the
	 * according values by evaluating the root {@link Condition}.
	 * 
	 * In case of parsing failure, an according exception is thrown.
	 * 
	 * ATTENTION: This method tests(!) for superfluous command line arguments
	 * being passed; e.g. command line arguments not being evaluated by any of
	 * the {@link Syntaxable} instance being traversed starting at the root
	 * {@link Condition} will be reported. Business logic therefore should
	 * invoke this root node's {@link #evalArgs(String[])} method instead of a
	 * {@link Condition}'s {@link Condition#parseArgs(String[])} method; as
	 * ignoring superfluous command line arguments will cause unexpected
	 * behavior from the point of view of the invoker.
	 * 
	 * 
	 * @param aArgs The command line arguments to be evaluated.
	 * 
	 * @return The list of evaluated command line arguments being instances of
	 *         the {@link Operand} interfaces or its sub-types.
	 * 
	 * @throws UnknownArgsException Thrown in case not one command line argument
	 *         matched regarding the provided args vs. the expected args.
	 * @throws AmbiguousArgsException Thrown in case at least one command line
	 *         argument is ambiguous regarding expected args vs. provided args.
	 * @throws SuperfluousArgsException Thrown in case there were arguments
	 *         found not being used (superfluous arguments).
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type
	 */
	List<? extends Operand<?>> evalArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException;

	/**
	 * Same as {@link #evalArgs(String[])} with the difference that the elements
	 * representing the arguments are passed as a list instead of an array.
	 * 
	 * @see #evalArgs(String[])
	 */
	default List<? extends Operand<?>> evalArgs( List<String> aArgs ) throws UnknownArgsException, AmbiguousArgsException, SuperfluousArgsException, ParseArgsException {
		return evalArgs( aArgs.toArray( new String[aArgs.size()] ) );
	}

	/**
	 * The root condition is the starting point node of a {@link Syntaxable}
	 * hierarchy to be traversed when determining the syntax for command line
	 * arguments or when evaluating the command line arguments.
	 * 
	 * @return The root {@link Condition} in which's syntax this parser is
	 *         based.
	 */
	Condition getRootCondition();

	void setBannerFont( Font aBannerFont );

	void setBannerFontPalette( char[] aColorPalette );

	/**
	 * Set the console's line break. A setting of null makes the instance use
	 * the {@link SystemUtility#getLineBreak()()} value.
	 * 
	 * @param aConsoleWidth The width to set.
	 */
	void setLineBreak( String aLineBreak );

	/**
	 * Set the console with. A setting of "-1" makes the instance use the
	 * {@link SystemUtility#getTerminalWidth()} value, i.e. the console width is
	 * set automatically to be the width of your terminal.
	 * 
	 * @param aConsoleWidth The width to set or -1 to let the parser
	 *        automatically determine the console width.
	 */
	void setConsoleWidth( int aConsoleWidth );

	/**
	 * Set the maximum console width to use in case the console width is greater
	 * than the maximum you want. This is most useful when the console width is
	 * determined automatically to be the width of your terminal.
	 * 
	 * @param aConsoleMaxWidth The maximum width to use.
	 */
	void setMaxConsoleWidth( int aMaxConsoleWidth );

	/**
	 * Set the standard out {@link PrintStream} and make other adjustments with
	 * the result (with regards to the builder pattern).
	 * 
	 * @param aStandardOut The standard out {@link PrintStream} to set.
	 */
	void setStandardOut( PrintStream aStandardOut );

	/**
	 * Set the error out {@link PrintStream} and make other adjustments with the
	 * result (with regards to the builder pattern).
	 * 
	 * @param aErrorOut The error out {@link PrintStream} to set.
	 */
	void setErrorOut( PrintStream aErrorOut );

	/**
	 * Set the {@link SyntaxNotation}.
	 * 
	 * @param aSyntaxNotation The {@link SyntaxNotation} to set.
	 */
	void setSyntaxNotation( SyntaxNotation aSyntaxNotation );

	/**
	 * Set the license note used by the {@link #printHelp()} method when writing
	 * out the licensing conditions.
	 * 
	 * @param aLicenseNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 */
	void setLicenseNote( String aLicenseNote );

	/**
	 * Set the usage label used by the {@link #printHelp()} method when writing
	 * out the syntax.
	 * 
	 * @param aUsageLabel The usage label printed out by the
	 *        {@link #printHelp()} method.
	 */
	void setUsageLabel( String aUsageLabel );

	/**
	 * Set the copyright note used by the {@link #printHelp()} method when
	 * writing out the copyright claim and make other adjustments with the
	 * result (with regards to the builder pattern).
	 * 
	 * @param aCopyrightNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 */
	void setCopyrightNote( String aCopyrightNote );

	/**
	 * Set the character to be used when printing a separator line with the
	 * {@link #printSeparatorLn()} method.
	 * 
	 * @param aSeparatorChar The character used by the
	 *        {@link #printSeparatorLn()} method when printing out the line of
	 *        characters..
	 */
	void setSeparatorChar( char aSeparatorChar );

	default ArgsParser withBannerFont( Font aBannerFont ) {
		setBannerFont( aBannerFont );
		return this;
	}

	default ArgsParser withBannerFontPalette( char[] aColorPalette ) {
		setBannerFontPalette( aColorPalette );
		return this;
	}

	/**
	 * Set the console's line break. A setting of null makes the instance use
	 * the {@link SystemUtility#getLineBreak()()} value.
	 * 
	 * @param aConsoleWidth The width to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withLineBreak( String aLineBreak ) {
		setLineBreak( aLineBreak );
		return this;
	}

	/**
	 * Set the console with. A setting of "-1" makes the instance use the
	 * {@link SystemUtility#getTerminalWidth()} value.
	 * 
	 * @param aConsoleWidth The width to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withConsoleWidth( int aConsoleWidth ) {
		setConsoleWidth( aConsoleWidth );
		return this;
	}

	/**
	 * Set the maximum console width to use in case the console width is greater
	 * than the maximum you want. This is most useful when the console width is
	 * determined automatically to be the width of your terminal.
	 * 
	 * @param aConsoleMaxWidth The maximum width to use.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withMaxConsoleWidth( int aMaxConsoleWidth ) {
		setMaxConsoleWidth( aMaxConsoleWidth );
		return this;
	}

	/**
	 * Set the standard out {@link PrintStream} and make other adjustments with
	 * the result (with regards to the builder pattern).
	 * 
	 * @param aStandardOut The standard out {@link PrintStream} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withStandardOut( PrintStream aStandardOut ) {
		setStandardOut( aStandardOut );
		return this;
	}

	/**
	 * Set the error out {@link PrintStream} and make other adjustments with the
	 * result (with regards to the builder pattern).
	 * 
	 * @param aErrorOut The error out {@link PrintStream} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withErrorOut( PrintStream aErrorOut ) {
		setErrorOut( aErrorOut );
		return this;
	}

	/**
	 * Set the {@link SyntaxNotation} and make other adjustments with the result
	 * (with regards to the builder pattern).
	 * 
	 * @param aSyntaxNotation The {@link SyntaxNotation} to set.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withSyntaxNotation( SyntaxNotation aSyntaxNotation ) {
		setSyntaxNotation( aSyntaxNotation );
		return this;
	}

	/**
	 * Set the license note used by the {@link #printHelp()} method when writing
	 * out the licensing conditions and make other adjustments with the result
	 * (with regards to the builder pattern).
	 * 
	 * @param aLicenseNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withLicenseNote( String aLicenseNote ) {
		setLicenseNote( aLicenseNote );
		return this;
	}

	/**
	 * Set the usage label used by the {@link #printHelp()} method when writing
	 * out the syntax and make other adjustments with the result (with regards
	 * to the builder pattern).
	 * 
	 * @param aUsageLabel The usage label printed out by the
	 *        {@link #printHelp()} method.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withUsageLabel( String aUsageLabel ) {
		setUsageLabel( aUsageLabel );
		return this;
	}

	/**
	 * Set the copyright note used by the {@link #printHelp()} method when
	 * writing out the copyright claim and make other adjustments with the
	 * result (with regards to the builder pattern).
	 * 
	 * @param aCopyrightNote The license note printed out by the
	 *        {@link #printHelp()} method.
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withCopyrightNote( String aCopyrightNote ) {
		setCopyrightNote( aCopyrightNote );
		return this;
	}

	/**
	 * Set the character to be used when printing a separator line with the
	 * {@link #printSeparatorLn()} method.
	 * 
	 * @param aSeparatorChar The character used by the
	 *        {@link #printSeparatorLn()} method when printing out the line of
	 *        characters..
	 * 
	 * @return This {@link ArgsParser} instance with regard to the builder
	 *         pattern.
	 */
	default ArgsParser withSeparatorChar( char aSeparatorChar ) {
		setSeparatorChar( aSeparatorChar );
		return this;
	}

	default ArgsParser withTitle( String aTitle ) {
		setTitle( aTitle );
		return this;
	}

	default ArgsParser withName( String aName ) {
		setName( aName );
		return this;
	}

	default ArgsParser withDescription( String aDescription ) {
		setDescription( aDescription );
		return this;
	}

	/**
	 * Prints the banner; the banner most probably is an ASCII art text block
	 * which's look depends strongly on the taste of the author implementing
	 * this interface.
	 */
	void printBanner();

	/**
	 * Prints the license note as specified by the
	 * {@link #withLicenseNote(String)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printLicenseNote();

	/**
	 * Prints the syntax as retrieved by the root {@link Condition#} with
	 * regards to the {@link SyntaxNotation} set by the
	 * {@link #withSyntaxNotation(SyntaxNotation)} method; making use of the
	 * usage label as passed by the {@link #withUsageLabel(String)} method.
	 */
	void printUsage();

	/**
	 * Prints the description as set by the {@link #withDescription(String)}
	 * method with regards to the console width as specified by the
	 * {@link #withConsoleWidth(int)} method.
	 */
	void printDescription();

	/**
	 * Prints the {@link Option}s (short- and the long-options), the
	 * {@link Switch}es and the {@link Operand} and their description with
	 * regards to the console width as specified by the
	 * {@link #withConsoleWidth(int)} method.
	 */
	void printOptions();

	/**
	 * Prints the copyright note as specified by the
	 * {@link #withCopyrightNote(String)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printCopyrightNote();

	/**
	 * Prints a separator line using the separator character as specified by the
	 * {@link #withSeparatorChar(char)} method; with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 */
	void printSeparatorLn();

	/**
	 * Prints the given line to standard out with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 * 
	 * @param aLine The line to be printed.
	 */
	void printLn( String aLine );

	/**
	 * Prints the given line to standard error with regards to to the console
	 * width as specified by the {@link #withConsoleWidth(int)} method.
	 * 
	 * @param aLine The line to be printed.
	 */
	void errorLn( String aLine );

	/**
	 * Prints an empty line / a line break.
	 */
	void printLn();

	/**
	 * Prints the help to the standard output specified by the
	 * {@link #withStandardOut(PrintStream)} method. This method can make use of
	 * the more atomic methods {@link #printBanner()}, {@link #printUsage()},
	 * {@link #printDescription()}, {@link #printOptions()} or
	 * {@link #printSeparatorLn()} to print a help text which's look depends
	 * strongly on the taste of the author implementing this interface. In case
	 * you dislike the implementing author's taste, feel free to overwrite this
	 * method and compose your own help text from the building blocks such as
	 * {@link #printBanner()}, {@link #printUsage()},
	 * {@link #printDescription()}, {@link #printOptions()} or
	 * {@link #printSeparatorLn()}
	 */
	void printHelp();

}
