// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console;

import org.refcodes.mixin.DescriptionAccessor;
import org.refcodes.mixin.TypeAccessor;
import org.refcodes.mixin.ValueAccessor;

/**
 * An {@link Operand} represents a value parsed from command line arguments. In
 * comparison to an {@link Option} an {@link Operand} just represents a plain
 * value without an identifier in front of it. An {@link Operand} has a state
 * which changes with each invocation of the {@link #parseArgs(String[])}
 * method.
 * <p>
 * It is recommended to put your {@link Operand} instance(s) at the end of your
 * top {@link Condition} to enforce it to be the last {@link Syntaxable}(s) when
 * parsing the command line arguments - this makes sure that any {@link Option}s
 * pick their option arguments so that the {@link Operand}(s) will correctly be
 * left over for parsing command line argument(s); the {@link Operand} will not
 * pick by mistake an option argument.
 * 
 */
public interface Operand<T> extends ValueAccessor<T>, DescriptionAccessor, TypeAccessor, Syntaxable, ArgsAccessor, Comparable<Operand<?>> {

	/**
	 * When being parsed via the {@link #parseArgs(String[])} method, then the
	 * {@link #getValue()} method returns the parsed value.
	 * 
	 * @return The parsed value when being parsed via the
	 *         {@link #parseArgs(String[])} method.
	 */
	@Override
	T getValue();

	/**
	 * When being parsed via the {@link #parseArgs(String[])} method, use this
	 * method to determine whether the {@link #getValue()} method contains
	 * parsed value.
	 * 
	 * @return The true in case there a value has been parsed via the
	 *         {@link #parseArgs(String[])} method.
	 */
	default boolean hasValue() {
		return getValue() != null;
	}

	/**
	 * When being parsed via the {@link #parseArgs(String[])} method, then the
	 * {@link #getArgs()} method returns the command line arguments representing
	 * this {@link Operand} instance (also including the short or the long
	 * options, see {@link Operand}).
	 * 
	 * @return The parsed command line arguments representing this
	 *         {@link Operand} instance.
	 */
	@Override
	String[] getArgs();

	/**
	 * Retrieves the name of the parameter value which can be the name of the
	 * operand ({@link Operand}) or the option argument ({@link Option}),
	 * depending on the sub-type inheriting from this interface. The parameter
	 * name is merely used for constructing the command line arguments syntax
	 * {@link String} via {@link #toSyntax(SyntaxNotation)} and the command line
	 * arguments detail description when creating a command line tool's help
	 * output.
	 * 
	 * @return The name of the parameter, r.g the name of the operand or the
	 *         name of the option argument.
	 */
	String getParameterName();

}
