// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console.impls;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.refcodes.console.Condition;
import org.refcodes.console.Operand;
import org.refcodes.console.SyntaxNotation;
import org.refcodes.console.Syntaxable;

/**
 * The {@link AbstractCondition} is an abstract implementation of an
 * {@link Condition} providing the boiler plate when implementing the
 * {@link Condition} interface.
 */
abstract public class AbstractCondition extends AbstractSyntaxable implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private List<Syntaxable> _children = new ArrayList<Syntaxable>();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	public AbstractCondition( Syntaxable... aElements ) {
		_children.addAll( Arrays.asList( aElements ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation ) {
		StringBuilder theBuilder = new StringBuilder( parseSyntax( aSyntaxNotation ) );
		if ( theBuilder.length() > 0 ) {
			theBuilder.insert( 0, "( " );
			theBuilder.append( " )" );
		}
		return theBuilder.toString();
	}

	@Override
	public void reset() {
		for ( Syntaxable eSyntaxable : _children ) {
			eSyntaxable.reset();
		}
	}

	@Override
	public String toString() {
		String theState = toState();
		// Remove annoying encapsulating parenthesis:
		if ( theState != null && theState.length() >= 4 && theState.startsWith( "( " ) && theState.endsWith( " )" ) ) {
			theState = theState.substring( 2 ).substring( 0, theState.length() - 4 );
		}
		return theState;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public List<? extends Operand<?>> toOperands() {
		List<Operand<?>> theList = new ArrayList<Operand<?>>();
		for ( Syntaxable eSyntaxable : getChildren() ) {
			if ( eSyntaxable instanceof Operand<?> ) {
				theList.add( (Operand<?>) eSyntaxable );
			}
			if ( eSyntaxable instanceof Condition ) {
				theList.addAll( ((Condition) eSyntaxable).toOperands() );
			}
		}
		return theList;
	}

	/**
	 *
	 */
	protected List<Syntaxable> getChildren() {
		return _children;
	}

	/**
	 *
	 */
	protected void addChild( Syntaxable aArgumentizer ) {
		_children.add( aArgumentizer );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
