// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console.impls;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.console.AmbiguousArgsException;
import org.refcodes.console.ConsoleUtility;
import org.refcodes.console.Operand;
import org.refcodes.console.Option;
import org.refcodes.console.ParseArgsException;
import org.refcodes.console.SyntaxNotation;
import org.refcodes.console.UnknownArgsException;
import org.refcodes.data.Prefixes;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;

/**
 * The {@link AbstractOperand} is an abstract implementation of an
 * {@link Operand} providing the boiler plate when implementing the
 * {@link Operand} interface.
 */
public abstract class AbstractOperand<T> extends AbstractSyntaxable implements Operand<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Class<T> _type;
	private String _parameterName;
	private String _description;
	private T _value = null;
	private String[] _args = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link Operand} with the given arguments.
	 * 
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aParameterName The {@link Operand}'s name, used for syntax
	 *        creation.
	 * @param aDescription A description without any line breaks.
	 */
	public AbstractOperand( Class<T> aType, String aParameterName, String aDescription ) {
		_type = aType;
		_parameterName = aParameterName;
		_description = aDescription;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public List<Operand<T>> parseArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException {
		int first = -1;
		for ( int i = aArgs.length - 1; i >= 0; i-- ) {
			if ( !ConsoleUtility.isOptionArgument( aArgs[i] ) ) {
				if ( first == -1 || first + -1 == i ) {
					first = i;
				}
			}
		}

		if ( first != -1 ) {
			List<Operand<T>> theList = new ArrayList<Operand<T>>();
			theList.add( this );
			setValue( toValue( aArgs[first] ) );
			setArgs( new String[] {
					aArgs[first]
			} );
			return theList;
		}

		throw new UnknownArgsException( aArgs, "Unable to determine an operand (not being prefixed with " + new VerboseTextBuilderImpl().withElements( Prefixes.ARG_OPTION ).toString() + ")." );
	}

	@Override
	public String getDescription() {
		return _description;
	}

	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation ) {
		return ConsoleUtility.toParameterSpec( this );
	}

	@Override
	public String getParameterName() {
		return _parameterName;
	}

	@Override
	public Class<?> getType() {
		return _type;
	}

	@Override
	public T getValue() {
		return _value;
	}

	@Override
	public String[] getArgs() {
		return _args;
	}

	@Override
	public void reset() {
		_args = null;
		_value = null;
	}

	@Override
	public String toState() {
		if ( _value != null ) {
			StringBuilder theBuilder = new StringBuilder();
			if ( getValue() instanceof String ) {
				theBuilder.append( '"' );
			}
			theBuilder.append( _value.toString() );
			if ( getValue() instanceof String ) {
				theBuilder.append( '"' );
			}
			return theBuilder.toString();
		}
		return "null";
	}

	// @Override
	// public List<Operand<T>> toOperands() {
	// List<Operand<T>> theOperands = new ArrayList<>();
	// theOperands.add( this );
	// return theOperands;
	// }

	@Override
	public String toString() {
		return toState();
	}

	@Override
	public int compareTo( Operand<?> obj ) {
		if ( obj instanceof Option<?> ) {
			if ( (this instanceof Option<?>) ) {
				String otherOption = ((Option<?>) obj).getShortOption() != null ? ((Option<?>) obj).getShortOption() : ((Option<?>) obj).getLongOption();
				String thisOption = ((Option<?>) this).getShortOption() != null ? ((Option<?>) this).getShortOption() : ((Option<?>) this).getLongOption();
				return thisOption.compareTo( otherOption );
			}
			if ( this instanceof Operand<?> ) { return 1; }
		}
		if ( obj instanceof Operand<?> ) {
			if ( this instanceof Option<?> ) { return -11; }
		}
		if ( getParameterName() != null && obj.getParameterName() != null ) { return getParameterName().compareTo( obj.getParameterName() ); }
		return toSyntax( null ).compareTo( obj.toSyntax( null ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sets the value for the {@link Operand} as parsed by the
	 * {@link #parseArgs(String[])} method.
	 * 
	 * @param aValue The value to be set for this {@link Operand}.
	 */
	protected void setValue( T aValue ) {
		_value = aValue;
	}

	/**
	 * Sets the command line argument(s) representing the {@link Operand} and
	 * its value as parsed by the {@link #parseArgs(String[])} method.
	 * 
	 * @param aArgs The command line arguments representing this {@link Operand}
	 *        with its value.
	 */
	protected void setArgs( String[] aArgs ) {
		_args = aArgs;
	}

	/**
	 * Double dispatch hook to be implemented by subclasses of the
	 * {@link AbstractOperand} for converting a command line argument to the
	 * required {@link Operand}'s type. In case conversion failed, then an
	 * according exception is to be thrown.
	 * 
	 * @param aArg The command line argument to be converted to an instance of
	 *        the given type T.
	 * 
	 * @return An instance of type T from the provided command line argument.
	 * 
	 * @throws ParseArgsException Thrown in case the provided command line
	 *         arguments do not respect the required syntax or cannot be
	 *         converted to the required type.
	 */
	abstract protected T toValue( String aArg ) throws ParseArgsException;

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
