// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console.impls;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.console.AmbiguousArgsException;
import org.refcodes.console.ConsoleUtility;
import org.refcodes.console.Operand;
import org.refcodes.console.Option;
import org.refcodes.console.ParseArgsException;
import org.refcodes.console.SyntaxNotation;
import org.refcodes.console.UnknownArgsException;
import org.refcodes.data.Prefix;
import org.refcodes.structure.Relation;

/**
 * The {@link AbstractOption} is an abstract implementation of an {@link Option}
 * providing the boiler plate when implementing the {@link Option} interface.
 */
public abstract class AbstractOption<T> extends AbstractOperand<T> implements Option<T> {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _shortOption;
	private String _longOption;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link Option} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aOptionArgName The option argument's name to be used when
	 *        constructing the syntax.
	 * @param aDescription A description without any line breaks.
	 */
	public AbstractOption( String aShortOption, String aLongOption, Class<T> aType, String aOptionArgName, String aDescription ) {
		super( aType, aOptionArgName, aDescription );
		if ( aShortOption != null && (!aShortOption.startsWith( Prefix.SHORT_ARG_OPTION.getPrefix() ) || aShortOption.startsWith( Prefix.LONG_ARG_OPTION.getPrefix() )) ) { throw new IllegalArgumentException( "Your short-option \"" + aShortOption + "\" must start exactly with \"" + Prefix.SHORT_ARG_OPTION.getPrefix() + "\"." ); }
		if ( aLongOption != null && !aLongOption.startsWith( Prefix.LONG_ARG_OPTION.getPrefix() ) ) { throw new IllegalArgumentException( "Your long-option \"" + aLongOption + "\" must start exactly with \"" + Prefix.SHORT_ARG_OPTION.getPrefix() + "\"." ); }
		_shortOption = aShortOption;
		_longOption = aLongOption;
	}

	/**
	 * Constructs a {@link Option} with the given arguments.
	 * 
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aType The type of the value returned by the {@link #getValue()}
	 *        method.
	 * @param aDescription A description without any line breaks.
	 */
	protected AbstractOption( String aShortOption, String aLongOption, Class<T> aType, String aDescription ) {
		super( aType, null, aDescription );
		_shortOption = aShortOption;
		_longOption = aLongOption;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public List<Operand<T>> parseArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException {
		Relation<String, String> theOptionArgument = ConsoleUtility.getOptionArgument( this, aArgs );
		if ( theOptionArgument != null ) {
			List<Operand<T>> theList = new ArrayList<Operand<T>>();
			theList.add( this );
			setArgs( new String[] {
					theOptionArgument.getKey(), theOptionArgument.getValue()
			} );
			setValue( toValue( theOptionArgument.getValue() ) );
			return theList;
		}
		if ( ConsoleUtility.contains( aArgs, getShortOption() ) ) { throw new ParseArgsException( aArgs, "Missing value; the short-option \"" + getShortOption() + "\" requires a value." ); }
		if ( ConsoleUtility.contains( aArgs, getLongOption() ) ) { throw new ParseArgsException( aArgs, "Missing value; the long-option \"" + getLongOption() + "\" requires a value." ); }
		throw new UnknownArgsException( aArgs, "Neither the short-option \"" + getShortOption() + "\" nor the long-option \"" + getLongOption() + "\"  was found in the command line arguments; at least one of them must be secified." );
	}

	@Override
	public String getShortOption() {
		return _shortOption;
	}

	@Override
	public String getLongOption() {
		return _longOption;
	}

	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation ) {
		return (getShortOption() != null ? getShortOption() : getLongOption()) + (getParameterName() != null ? (" " + ConsoleUtility.toParameterSpec( this )) : "");
	}

	@Override
	public String toState() {
		String theSwitch = null;
		if ( getShortOption() != null ) {
			theSwitch = getShortOption();
		}
		else if ( getLongOption() != null ) { return theSwitch = getLongOption(); }
		if ( theSwitch != null ) {
			StringBuilder theBuilder = new StringBuilder();
			theBuilder.append( theSwitch );
			theBuilder.append( ":=" );
			if ( getValue() != null ) {
				if ( getValue() instanceof String ) {
					theBuilder.append( '"' );
				}
				theBuilder.append( getValue().toString() );
				if ( getValue() instanceof String ) {
					theBuilder.append( '"' );
				}
			}
			else {
				theBuilder.append( "null" );
			}
			return theBuilder.toString();
		}

		return "null";

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	protected void setValue( T aValue ) {
		super.setValue( aValue );
	}

	@Override
	protected void setArgs( String[] aArgs ) {
		super.setArgs( aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
