// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console.impls;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.console.AmbiguousArgsException;
import org.refcodes.console.Condition;
import org.refcodes.console.ConsoleUtility;
import org.refcodes.console.Operand;
import org.refcodes.console.ParseArgsException;
import org.refcodes.console.SyntaxNotation;
import org.refcodes.console.Syntaxable;
import org.refcodes.console.UnknownArgsException;

/**
 * An {@link AndConditionImpl} represents a list of {@link Syntaxable} instances
 * of which all are be parsed successfully when the {@link Syntaxable}s'
 * {@link Syntaxable#parseArgs(String[])} methods are invoked. The command line
 * arguments syntax -a & -b & -c specifies that all "-a", "-b" and "-c" must be
 * set. In case at least one is not set, then the {@link AndConditionImpl} will
 * terminate the {@link #parseArgs(String[])} method with an exception.
 */
public class AndConditionImpl extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String RIGHT_TAG = " )";
	private static final String LEFT_TAG = "( ";
	private static final char GUN_POSIX_AND = ' ';
	private static final String LOGICAL_AND = " & ";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	public AndConditionImpl( Syntaxable... aArgs ) {
		super( aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException {
		List<Operand<?>> theResult = new ArrayList<Operand<?>>();
		List<Operand<?>> eOperands;
		for ( Syntaxable eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs );
			}
			catch ( UnknownArgsException exc ) {
				throw new UnknownArgsException( exc.getArgs(), "At least one mandatory operand (option) did not match the provided command line arguments; though all must match.", exc );
			}
			catch ( AmbiguousArgsException exc ) {
				throw new AmbiguousArgsException( exc.getArgs(), "At least one mandatory operand (option) did not match the provided command line arguments; though all must match.", exc );
			}
			if ( eOperands != null ) {
				theResult.addAll( (List<? extends Operand<?>>) eOperands );
			}
			aArgs = ConsoleUtility.toDiff( aArgs, eOperands );
		}
		return theResult;
	}

	@Override
	public String parseSyntax( SyntaxNotation aSyntaxNotation ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eArgumentizer : getChildren() ) {
			if ( theBuilder.length() != 0 ) {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_AND );
				}
				else {
					theBuilder.append( GUN_POSIX_AND );
				}
			}
			theBuilder.append( eArgumentizer.toSyntax( aSyntaxNotation ) );
		}
		return theBuilder.toString();
	}

	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eArgumentizer : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( LOGICAL_AND );
			}
			theBuilder.append( eArgumentizer.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
