// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.console.impls;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.refcodes.console.AmbiguousArgsException;
import org.refcodes.console.Condition;
import org.refcodes.console.ConsoleUtility;
import org.refcodes.console.Operand;
import org.refcodes.console.Option;
import org.refcodes.console.ParseArgsException;
import org.refcodes.console.SyntaxNotation;
import org.refcodes.console.Syntaxable;
import org.refcodes.console.UnknownArgsException;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;

/**
 * An {@link XorConditionImpl} represents a list of {@link Syntaxable} instances
 * of which only one is allowed to be parsed successfully when the
 * {@link Syntaxable}s' {@link Syntaxable#parseArgs(String[])} methods are
 * invoked. The command line arguments syntax { -a ^ -b ^ -c } specifies that
 * only "-a", only "-b" or only "-c" must be set. In case more then one is set
 * or none, then the {@link XorConditionImpl} will terminate the
 * {@link #parseArgs(String[])} method with an exception.
 */
public class XorConditionImpl extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String RIGHT_TAG = " )";
	private static final String LEFT_TAG = "( ";
	private static final String GUN_POSIX_XOR = " | ";
	private static final String LOGICAL_XOR = " ^ ";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	public XorConditionImpl( Syntaxable... aArgs ) {
		super( aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs ) throws UnknownArgsException, AmbiguousArgsException, ParseArgsException {
		List<Condition> theOptionals = new ArrayList<Condition>();
		List<Operand<?>> theResult = null;
		Syntaxable theResultSyntaxable = null;
		List<Operand<?>> eOperands;
		for ( Syntaxable eChild : getChildren() ) {
			if ( eChild instanceof OptionalConditionImpl ) theOptionals.add( (OptionalConditionImpl) eChild );
			eOperands = null;
			try {
				if ( theResult != null && eChild instanceof Operand && theResultSyntaxable instanceof Option ) {
					eOperands = (List<Operand<?>>) eChild.parseArgs( ConsoleUtility.toDiff( aArgs, theResult ) );
				}
				else {
					eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs );
				}
				if ( eOperands != null && eOperands.isEmpty() ) eOperands = null;
			}
			catch ( UnknownArgsException | AmbiguousArgsException exc ) {}
			if ( theResult != null && eOperands != null ) {
				throw new AmbiguousArgsException( aArgs, "More than one exclusive operand (option) matched the provided command line arguments; though exactly one must match: " + new VerboseTextBuilderImpl().withElements( ConsoleUtility.toArgs( theResult, eOperands ) ).toString() );
			}
			if ( eOperands != null && !eOperands.isEmpty() && theResult == null ) {
				theResult = eOperands;
				theResultSyntaxable = eChild;
			}

		}
		if ( theResult != null ) {
			return theResult;
		}
		if ( theOptionals.size() == 1 ) {
			return Collections.EMPTY_LIST;
		}
		// Get the parameters from the optionals to create a good message |-->
		else if ( theOptionals.size() > 1 ) {
			throw new AmbiguousArgsException( aArgs, "None optional operand (option) matched the provided command line arguments; causing exclusive contextual ambiguity: " + new VerboseTextBuilderImpl().withElements( toSpec( theOptionals ) ).toString() );
		}
		// <--| Get the parameters from the optionals to create a good message
		throw new UnknownArgsException( aArgs, "Not one exclusive operand (option) matched the provided command line arguments; though exactly one must match." );
	}

	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation ) {
		StringBuilder theBuilder = new StringBuilder( parseSyntax( aSyntaxNotation ) );
		if ( theBuilder.length() > 0 ) {
			if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
				theBuilder.insert( 0, LEFT_TAG );
			}
			else {
				theBuilder.insert( 0, "{ " );
			}
			if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
				theBuilder.append( RIGHT_TAG );
			}
			else {
				theBuilder.append( " }" );
			}
		}
		return theBuilder.toString();
	}

	@Override
	public String parseSyntax( SyntaxNotation aSyntaxNotation ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eArgumentizer : getChildren() ) {
			if ( theBuilder.length() != 0 ) {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_XOR );
				}
				else {
					theBuilder.append( GUN_POSIX_XOR );
				}
			}
			theBuilder.append( eArgumentizer.toSyntax( aSyntaxNotation ) );
		}
		return theBuilder.toString();
	}

	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eArgumentizer : getChildren() ) {

			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( LOGICAL_XOR );
			}
			theBuilder.append( eArgumentizer.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static List<String> toSpec( List<Condition> aConditions ) {
		List<String> theParameters = new ArrayList<>();
		for ( Condition eCondition : aConditions ) {
			for ( Operand<?> eOperand : eCondition.toOperands() ) {
				theParameters.add( ConsoleUtility.toSpec( eOperand ) );
			}
		}
		return theParameters;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
