// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

/**
 * <h2>Getting started:</h2>
 *
 * Consider you have a tool called "foo-bar" to be invoked with the below
 * allowed argument combinations (syntax):
 * <p>
 * foo-bar [{ -a | -d }] -f &lt;file&gt;
 * <p>
 * "foo-bar" can be invoked either with an optional "-a" or with an optional
 * "-d" switch, but not both of them at the same time, and a file
 * "-f &lt;file&gt;" must be provided, else the passed arguments are rejected as
 * not being valid.
 * <p>
 * Valid arguments would be:
 * <ul>
 * <li>foo-bar -f someFile
 * <li>foo-bar -d -f anyFile
 * <li>foo-bar -f otherFile -a
 * <li>foo-bar -a -f otherFile
 * </ul>
 * Invalid arguments would be:
 * <ul>
 * <li>foo-bar -f someFile -b
 * <li>foo-bar -a someFile -f
 * <li>foo-bar -a -d -f anyFile
 * <li>foo-bar -a -x -f otherFile
 * </ul>
 * This means that additional switches not supported are not valid. The parser
 * detects such situations and you can print out a help message in such cases.
 *
 * <h2>Construct your parser:</h2>
 *
 * First build your syntax using {@link org.refcodes.console.Switch}es ,
 * {@link org.refcodes.console.Option}s and
 * {@link org.refcodes.console.Condition}s. You actually build the syntax tree
 * for your command line tool's supported arguments:
 * 
 * <code>
 * Option<String> theFile = new StringOptionImpl( "-f", "--file", "file", "A file" );
 * Switch theAdd = new SwitchImpl( "-a", null, "Add the specified file" );
 * Switch theDelete = new SwitchImpl( "-d", null, "Delete the specified file" );
 * Condition theXor = new XorConditionImpl( theAdd, theDelete );
 * Syntaxable theOptional = new OptionalImpl( theXor );
 * Condition theAnd = new AndConditionImpl( theOptional, theFile );
 * 
 * ArgsParser theArgsParser = new ArgsParserImpl( theAnd );
 * theArgsParser.printUsage();
 * // theArgsParser.printHelp();
 * </code>
 * 
 * <h2>Using syntactic sugar</h2>
 * 
 * The
 * [`TinyRestfulServer`](https://bitbucket.org/refcodes/funcodes-tinyrestful/raw
 * /master/src/main/java/club/funcodes/tinyrestful/TinyRestfulServer.java) demo
 * application uses `syntactic sugar` for setting up the command line arguments
 * parser:
 * 
 * <code>
 * import static org.refcodes.console.ConsoleSugar.*;
 * ...
 * 	public static void main( String args[] ) {
 * 
 * 		Option<Integer> theWidth = intOption( "-w", "--width", "width", "Sets the console width" );
 * 		Option<Integer> thePort = intOption( "-p", "--port", "port", "Sets the port for the server" );
 * 		Option<Integer> theMaxConns = intOption( "-c", "--connections", "connections", "Sets the number of max. connections" );
 * 		Option<String> theUsername = stringOption( "-u", "--user", "username", "The username for HTTP basic authentication" );
 * 		Option<String> theSecret = stringOption( "-s", "--secret", "secret", "The password for HTTP basic authentication" );
 * 
 * 		Switch theHelp = helpSwitch( "Shows this help" );
 * 		Condition theRootCondition = xor( 
 * 			and( 
 * 				thePort, optional( theMaxConns ), optional( and( theUsername, theSecret ) ), optional( theWidth )
 * 			),
 * 			theHelp
 * 		);
 * 		ArgsParser theArgsParser = new ArgsParserImpl( theRootCondition );
 * 		theArgsParser.withSyntaxNotation( SyntaxNotation.REFCODES );
 * 		theArgsParser.withName( "TinyRestful" ).withTitle( "TINYRESTFUL" ).withCopyrightNote( "Copyright (c) by FUNCODES.CLUB, Munich, Germany." ).withLicenseNote( "Licensed under GNU General Public License, v3.0 and Apache License, v2.0" );
 * 		theArgsParser.withBannerFont( new FontImpl( FontType.DIALOG, FontStyle.BOLD, 14 ) ).withBannerFontPalette( AsciiColorPalette.MAX_LEVEL_GRAY.getPalette() );
 * 		theArgsParser.setDescription( "Tiny evil RESTful server. TinyRestfulServer makes heavy use of the REFCODES.ORG artifacts found together with the FUNCODES.CLUB sources at <http://bitbucket.org/refcodes>." );
 * 		List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
 * 		...
 * 	}
 * ...
 * </code>
 * 
 * Most obvious is the missing <code>new</code> statement for instantiating the parts of
 * your command line parser as this is done by the statically imported methods.
 * 
 * ### Under the hood
 * 
 * As seen above, you pass your root {@link org.refcodes.console.Condition} to
 * the {@link org.refcodes.console.ArgsParser} which then already can print out
 * your command line tool's usage string:
 *
 * <h2>Test (invoke) your parser:</h2>
 *
 * In real live you would pass your main-method's args[] array to the parser.
 * Now just for a test-run, pass a {@link java.lang.String} array to your parser
 * and let it parse it:
 * 
 * <code>
 * String[] args = new String[] {
 *	"-f", "someFile", "-d"
 * };
 * List<? extends Operand<?>> theResult = theArgsParser.evalArgs( args );
 * File theConfigFile = new File( theFile.getValue() );
 * </code>
 * 
 * Now the leaves of your syntax tree are filled with the argument's values
 * according to the syntax you have been setting up: Your
 * {@link org.refcodes.console.impls.StringOptionImpl} instance aliased
 * "theFile" now contains the value "someFile".
 * <p>
 * The "theResult" contains the parsed arguments for you to use in your business
 * logic.
 * <p>
 * In case of argument rejection, a sub-type of the
 * {@link org.refcodes.console.ArgsMismatchException} is being thrown; actually
 * one of the exceptions {@link org.refcodes.console.UnknownArgsException},
 * {@link org.refcodes.console.AmbiguousArgsException},
 * {@link org.refcodes.console.SuperfluousArgsException} or
 * {@link org.refcodes.console.ParseArgsException} is being thrown, according to
 * the cause of the rejection. So you can either catch the
 * {@link org.refcodes.console.ArgsMismatchException} or, if you need more
 * details on the cause, the other sub-exceptions.
 *
 * <h2>{@link org.refcodes.console.Syntaxable}:</h2>
 *
 * {@link Syntaxable} defines the methods at least required when building a
 * command line arguments syntax tree for traversing the syntax tree; either for
 * parsing command line arguments or for constructing the command line arguments
 * syntax.
 * <p>
 * By providing various implementations of the {@link Syntaxable}'s subclasses
 * such as {@link Operand}, {@link Option} or {@link Condition}, a command line
 * arguments syntax tree can be constructed. This syntax tree can be use to
 * create a human readable (verbose) command line arguments syntax and to parse
 * an array of command line arguments for determining the {@link Operand}s', the
 * {@link Switch}es' or the {@link Option}s' values.
 *
 * <h2>{@link org.refcodes.console.Operand}:</h2>
 *
 * An {@link Operand} represents a value parsed from command line arguments. An
 * {@link Operand} has a state which changes with each invocation of the
 * {@link #parseArgs(String[])} method.
 * <p>
 * It is recommended to put your {@link Operand} instance(s) at the end of your
 * top {@link Condition} to enforce it to be the last {@link Syntaxable}(s) when
 * parsing the command line arguments - this makes sure that any {@link Option}s
 * pick their option arguments so that the {@link Operand}(s) will correctly be
 * left over for parsing command line argument(s); the {@link Operand} will not
 * pick by mistake an {@link org.refcodes.console.Option} argument.
 *
 * <h2>{@link org.refcodes.console.Option}:</h2>
 *
 * An {@link Option} represents a command line option with the according
 * option's value. An {@link Option} can be seen as a key/value(s) pair defined
 * in the command line arguments parsed via the {@link #parseArgs(String[])}
 * method.
 * <p>
 * An {@link Option} has a state which changes with each invocation of the
 * {@link #parseArgs(String[])} method.
 *
 * <h2>{@link org.refcodes.console.Switch}:</h2>
 *
 * A {@link Switch} is an {@link Option} with a {@link Boolean} state. Usually
 * switches are just set or omitted in the command line arguments with no value
 * provided; former representing a <code>true</code> status and latter
 * representing a <code>false</code> status.
 *
 * <h2>{@link org.refcodes.console.Condition}:</h2>
 *
 * The {@link Condition} interface represents a node in the command line
 * arguments syntax tree; simply extending the {@link Syntaxable} interface and
 * adding the functionality of providing access to the added {@link Operand}s
 * (leafs). In future extensions, a {@link Condition} might provide access to
 * the child {@link Syntaxable} elements contained in a {@link Condition}
 * instance. As of the current findings, access to the children of the
 * {@link Condition} node is not required and would make the interface
 * unnecessarily complicated.
 * <p>
 *
 * @see org.refcodes.console.impls.StringOptionImpl
 * @see org.refcodes.console.impls.SwitchImpl
 * @see org.refcodes.console.impls.StringOperandImpl
 * @see org.refcodes.console.impls.XorConditionImpl
 * @see org.refcodes.console.impls.AndConditionImpl
 * @see org.refcodes.console.impls.OptionalConditionImpl
 */
package org.refcodes.console;