// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.controlflow;

import static org.junit.jupiter.api.Assertions.*;

import java.lang.Thread.State;

import org.junit.jupiter.api.Test;

/**
 * Tests the {@link ExceptionWatchdog}.
 */
public class ExceptionWatchdogTest {

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	/**
	 * Test single exception watchdog.
	 */
	@Test
	public void testSingleExceptionWatchdog() {
		final ExceptionWatchdog<Exception> theExceptionWatchdog = new ExceptionWatchdog<Exception>();
		AwaitExceptionThread t1 = new AwaitExceptionThread( theExceptionWatchdog );
		t1.start();
		pause( 100 );
		if ( IS_LOG_TEST_ENABLED ) System.out.println( "Calling #nextException( ... )" );
		theExceptionWatchdog.throwException( new Exception( "Catch me if you can!" ) );
		pause( 100 );
		theExceptionWatchdog.releaseAll();
		pause( 100 );
		assertEquals( 1, t1.getExceptionCount() );
		pause( 100 );
		assertFalse( t1.isAlive() );
	}

	/**
	 * Test two exception watchdog.
	 */
	@Test
	public void testTwoExceptionWatchdog() {
		final ExceptionWatchdog<Exception> theExceptionWatchdog = new ExceptionWatchdog<Exception>();
		AwaitExceptionThread t1 = new AwaitExceptionThread( theExceptionWatchdog );
		t1.start();
		pause( 100 );
		if ( IS_LOG_TEST_ENABLED ) System.out.println( "Calling #nextException( ... )" );
		theExceptionWatchdog.throwException( new Exception( "Catch me if you can!" ) );
		pause( 100 );
		theExceptionWatchdog.throwException( new Exception( "Catch as catch can!" ) );
		pause( 100 );
		theExceptionWatchdog.releaseAll();
		assertEquals( t1.getExceptionCount(), 2 );
		pause( 100 );
		assertFalse( t1.isAlive() );
	}

	/**
	 * Test multi exception watchdog.
	 */
	@Test
	public void testMultiExceptionWatchdog() {
		final ExceptionWatchdog<Exception> theExceptionWatchdog = new ExceptionWatchdog<Exception>();
		AwaitExceptionThread[] t = new AwaitExceptionThread[10];
		for ( int i = 0; i < 10; i++ ) {
			t[i] = new AwaitExceptionThread( theExceptionWatchdog );
			t[i].start();
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println( "Calling #throwException( ... )" );
		waitForThreads( t );
		theExceptionWatchdog.throwException( new Exception( "Catch me if you can!" ) );
		waitForThreads( t );
		theExceptionWatchdog.throwException( new Exception( "Catch as catch can!" ) );
		waitForThreads( t );
		theExceptionWatchdog.releaseAll();
		for ( int i = 0; i < t.length; i++ ) {
			assertEquals( 2, t[i].getExceptionCount() );
		}
		pause( 500 );
		for ( int i = 0; i < t.length; i++ ) {
			assertFalse( t[i].isAlive() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Waits for the threads to settle in state WAITING.
	 *
	 * @param t the t
	 */
	private void waitForThreads( AwaitExceptionThread[] t ) {
		for ( int i = 0; i < 10; i++ ) {
			while ( t[i].getState() != State.WAITING ) {
				if ( IS_LOG_TEST_ENABLED ) System.out.println( t[i].getName() + " = " + t[i].getState() );
				pause( 100 );
			}
		}
	}

	/**
	 * Pauses for 100ms to give threads time to settle themselves.
	 *
	 * @param aTimeout the timeout
	 */
	private void pause( int aTimeout ) {
		try {
			Thread.sleep( aTimeout );
		}
		catch ( InterruptedException ignored ) {}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class AwaitExceptionThread.
	 */
	private class AwaitExceptionThread extends Thread {

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private ExceptionWatchdog<Exception> _exceptionWatchdog;
		private int _exceptionCount = 0;
		private boolean _isStarted = false;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new await exception thread.
		 *
		 * @param aExceptionWatchdog the exception watchdog
		 */
		public AwaitExceptionThread( ExceptionWatchdog<Exception> aExceptionWatchdog ) {
			_exceptionWatchdog = aExceptionWatchdog;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		/**
		 * Start.
		 */
		// /////////////////////////////////////////////////////////////////////
		@Override
		public void start() {
			super.start();
			RetryCounter theRetry = new RetryCounter( 10, 100 );
			while ( theRetry.hasNextRetry() && !_isStarted ) theRetry.nextRetry();
		}

		/**
		 * Run.
		 */
		@Override
		public void run() {
			if ( IS_LOG_TEST_ENABLED ) System.out.println( "Thread [" + Thread.currentThread().getId() + "]: Thread started ..." );
			Exception theException;
			do {
				theException = null;
				try {
					_isStarted = true;
					if ( IS_LOG_TEST_ENABLED ) System.out.println( "Thread [" + Thread.currentThread().getId() + "]: PRE-catch ..." );
					_exceptionWatchdog.catchException();
				}
				catch ( Exception e ) {
					if ( IS_LOG_TEST_ENABLED ) System.out.println( "Thread [" + Thread.currentThread().getId() + "]: POST-catch." );
					theException = e;
					_exceptionCount++;
					if ( IS_LOG_TEST_ENABLED ) System.out.println( "Thread [" + Thread.currentThread().getId() + "]: Caught an exception \"" + e.getMessage() + "\" <" + _exceptionCount + ">." );
				}
			} while ( theException != null );
			if ( IS_LOG_TEST_ENABLED ) System.out.println( "Thread [" + Thread.currentThread().getId() + "]: Thread released <" + _exceptionCount + ">." );
		}

		/**
		 * Gets the exception count.
		 *
		 * @return the exception count
		 */
		public int getExceptionCount() {
			return _exceptionCount;
		}
	}
}
