// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.criteria;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Base class with the base {@link org.refcodes.criteria.SingleCriteriaNode}
 * functionality provided for {@link org.refcodes.criteria.SingleCriteriaNode}
 * implementations.
 *
 * @author steiner
 * 
 * @version $Id: $Id
 */
public abstract class AbstractSingleCriteriaNode extends AbstractCriteria implements SingleCriteriaNode {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private List<Criteria> _children;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link org.refcodes.criteria.SingleCriteriaNode} with the
	 * given name.
	 *
	 * @param aName The name of the
	 *        {@link org.refcodes.criteria.SingleCriteriaNode}.
	 */
	public AbstractSingleCriteriaNode( String aName ) {
		super( aName );
	}

	/**
	 * Constructs a {@link org.refcodes.criteria.SingleCriteriaNode} with the
	 * given name and the provided {@link org.refcodes.criteria.Criteria} child.
	 *
	 * @param aName The name of the
	 *        {@link org.refcodes.criteria.SingleCriteriaNode}.
	 * @param aChild The {@link org.refcodes.criteria.Criteria} child to be
	 *        contained in the {@link org.refcodes.criteria.SingleCriteriaNode}.
	 */
	public AbstractSingleCriteriaNode( String aName, Criteria aChild ) {
		super( aName );
		setChild( aChild );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Criteria getChild() {
		if ( _children.size() == 0 ) {
			return null;
		}
		return _children.get( 0 );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setChild( Criteria aChild ) {
		List<Criteria> theChildren = new ArrayList<Criteria>();
		if ( aChild != null ) {
			theChildren.add( aChild );
		}
		else {
			theChildren.clear();
		}
		_children = Collections.unmodifiableList( theChildren );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addChild( Criteria aCriteria ) throws IllegalStateException, IllegalArgumentException {
		if ( getChildren().size() > 0 ) {
			throw new IllegalStateException( "A 'single criteria' node must not have more than one criteria, the 'single criteria' node already contains <" + getChildren().size() + "> criteria children." );
		}
		setChild( aCriteria );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Criteria> getChildren() {
		return _children;
	}
}
