// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.criteria;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions of {@link org.refcodes.criteria.Criteria}
 * trees: ... Criteria theCriteria = and( or( equalWith( "City", "Berlin" ),
 * equalWith( "City", "Munich" ) ), equalWith( "Surname", "Miller" ) ); ...
 *
 * @author steiner
 * 
 * @version $Id: $Id
 */
public class CriteriaSugar {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prevent instantiation as this is a utility class with static methods.
	 */
	private CriteriaSugar() {}

	// /////////////////////////////////////////////////////////////////////////
	// IMPORT STATICS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates an AND {@link org.refcodes.criteria.CriteriaNode} containing the
	 * provided {@link org.refcodes.criteria.Criteria} children.
	 *
	 * @param aChildren The {@link org.refcodes.criteria.Criteria} children to
	 *        be contained in the AND
	 *        {@link org.refcodes.criteria.CriteriaNode}.
	 * 
	 * @return The AND {@link org.refcodes.criteria.CriteriaNode}.
	 */
	public static AndCriteria and( Criteria... aChildren ) {
		return new AndCriteriaImpl( aChildren );
	}

	/**
	 * Creates an EQUAL WITH ("=") {@link org.refcodes.criteria.CriteriaLeaf}
	 * expression relating to the given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The EQUAL WITH {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> EqualWithCriteria<T> equalWith( String aKey, T aValue ) {
		return new EqualWithCriteriaImpl<T>( aKey, aValue );
	}

	/**
	 * Creates an GREATER OR EQUAL THAN ("&gt;=")
	 * {@link org.refcodes.criteria.CriteriaLeaf} expression relating to the
	 * given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The GREATER OR EQUAL THAN
	 *         {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> GreaterOrEqualThanCriteria<T> greaterOrEqualThan( String aKey, T aValue ) {
		return new GreaterOrEqualThanCriteriaImpl<T>( aKey, aValue );
	}

	/**
	 * Creates an GREATER THAN ("&gt;")
	 * {@link org.refcodes.criteria.CriteriaLeaf} expression relating to the
	 * given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The GREATER THAN {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> GreaterThanCriteria<T> greaterThan( String aKey, T aValue ) {
		return new GreaterThanCriteriaImpl<T>( aKey, aValue );
	}

	/**
	 * Creates an INTERSECT {@link org.refcodes.criteria.CriteriaNode}
	 * containing the provided {@link org.refcodes.criteria.Criteria} children.
	 *
	 * @param aChildren The {@link org.refcodes.criteria.Criteria} children to
	 *        be contained in the INTERSECT
	 *        {@link org.refcodes.criteria.CriteriaNode}.
	 * 
	 * @return The INTERSECT {@link org.refcodes.criteria.CriteriaNode}.
	 */
	public static IntersectWithCriteria intersectWith( Criteria... aChildren ) {
		return new IntersectWithCriteriaImpl( aChildren );
	}

	/**
	 * Creates an LESS OR EQUAL THAN ("&lt;=")
	 * {@link org.refcodes.criteria.CriteriaLeaf} expression relating to the
	 * given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The LESS OR EQUAL THAN
	 *         {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> LessOrEqualThanCriteria<T> lessOrEqualThan( String aKey, T aValue ) {
		return new LessOrEqualThanCriteriaImpl<T>( aKey, aValue );
	}

	/**
	 * Creates an LESS THAN {@link org.refcodes.criteria.CriteriaLeaf}
	 * expression relating to the given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The LESS THAN {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> LessThanCriteria<T> lessThan( String aKey, T aValue ) {
		return new LessThanCriteriaImpl<T>( aKey, aValue );
	}

	/**
	 * Creates a NOT {@link org.refcodes.criteria.CriteriaNode} containing the
	 * provided {@link org.refcodes.criteria.Criteria} child.
	 *
	 * @param aCriteria The {@link org.refcodes.criteria.Criteria} child to be
	 *        contained in the NOT {@link org.refcodes.criteria.CriteriaNode}.
	 * 
	 * @return The NOT {@link org.refcodes.criteria.CriteriaNode}.
	 */
	public static NotCriteria not( Criteria aCriteria ) {
		return new NotCriteriaImpl( aCriteria );
	}

	/**
	 * Creates an OR {@link org.refcodes.criteria.CriteriaNode} containing the
	 * provided {@link org.refcodes.criteria.Criteria} children.
	 *
	 * @param aChildren The {@link org.refcodes.criteria.Criteria} children to
	 *        be contained in the OR {@link org.refcodes.criteria.CriteriaNode}.
	 * 
	 * @return The OR {@link org.refcodes.criteria.CriteriaNode}.
	 */
	public static OrCriteria or( Criteria... aChildren ) {
		return new OrCriteriaImpl( aChildren );
	}

	/**
	 * Creates a NOT EQUAL WITH ("&lt;&gt;")
	 * {@link org.refcodes.criteria.CriteriaLeaf} expression relating to the
	 * given key and the given reference value.
	 *
	 * @param <T> The type of the {@link org.refcodes.criteria.CriteriaLeaf}'s
	 *        value.
	 * @param aKey The key on which the reference value is to be applied.
	 * @param aValue The reference value to be applied on the key.
	 * 
	 * @return The NOT EQUAL WITH {@link org.refcodes.criteria.CriteriaLeaf}.
	 */
	public static <T> NotEqualWithCriteria<T> notEqualWith( String aKey, T aValue ) {
		return new NotEqualWithCriteriaImpl<T>( aKey, aValue );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
	// The actual Criteria implementations do not provide more than an instance
	// being of the according Criteria type and the defined name for the
	// Criteria. There is nothing more implementation specific. For using the
	// Criteria tree merely the type is important, therefore the implementations
	// are "hidden" away from you.
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Implementation of the according {@link Criteria}.
	 */
	private static class AndCriteriaImpl extends AbstractCriteriaNode implements AndCriteria {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public AndCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Constructor setting the children {@link Criteria} of the
		 * {@link Criteria}.
		 * 
		 * @param aChildren The children {@link Criteria} instances to be set.
		 */
		public AndCriteriaImpl( Criteria... aChildren ) {
			super( NAME, aChildren );
		}

	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class EqualWithCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements EqualWithCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public EqualWithCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public EqualWithCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class GreaterOrEqualThanCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements GreaterOrEqualThanCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public GreaterOrEqualThanCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public GreaterOrEqualThanCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 */
	private static class IntersectWithCriteriaImpl extends AbstractCriteriaNode implements IntersectWithCriteria {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public IntersectWithCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Constructor setting the children {@link Criteria} of the
		 * {@link Criteria}.
		 * 
		 * @param aChildren The children {@link Criteria} instances to be set.
		 */
		public IntersectWithCriteriaImpl( Criteria... aChildren ) {
			super( NAME, aChildren );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class GreaterThanCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements GreaterThanCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public GreaterThanCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public GreaterThanCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 */
	private static class OrCriteriaImpl extends AbstractCriteriaNode implements OrCriteria {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public OrCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Constructor setting the children {@link Criteria} of the
		 * {@link Criteria}.
		 * 
		 * @param aChildren The children {@link Criteria} instances to be set.
		 */
		public OrCriteriaImpl( Criteria... aChildren ) {
			super( NAME, aChildren );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class NotEqualWithCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements NotEqualWithCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public NotEqualWithCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public NotEqualWithCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 */
	private static class NotCriteriaImpl extends AbstractSingleCriteriaNode implements NotCriteria {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public NotCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Constructor setting the child {@link Criteria} for the
		 * {@link Criteria}.
		 * 
		 * @param aChild The child {@link Criteria} instance to be set.
		 */
		public NotCriteriaImpl( Criteria aChild ) {
			super( NAME, aChild );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class LessThanCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements LessThanCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public LessThanCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public LessThanCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}

	/**
	 * Implementation of the according {@link Criteria}.
	 *
	 * @param <T> the generic type
	 */
	private static class LessOrEqualThanCriteriaImpl<T> extends AbstractCriteriaLeaf<T> implements LessOrEqualThanCriteria<T> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Empty constructor setting the name for the {@link Criteria}.
		 */
		public LessOrEqualThanCriteriaImpl() {
			super( NAME );
		}

		/**
		 * Sets the key and the according value for the {@link Criteria},.
		 *
		 * @param aKey The key for which the {@link Criteria} is responsible.
		 * @param aValue The value referenced by this {@link Criteria}.
		 */
		public LessOrEqualThanCriteriaImpl( String aKey, T aValue ) {
			super( NAME, aKey, aValue );
		}
	}
}
