// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.data;

import org.refcodes.mixin.KeyAccessor;
import org.refcodes.mixin.ValueAccessor.ValueProperty;

/**
 * {@link EnvironmentProperty} for retrieving some common environment variables
 * with ease.
 */
public enum EnvironmentProperty implements KeyAccessor<String>, ValueProperty<String> {

	/**
	 * The console height environment variable.
	 */
	TERMINAL_HEIGHT("LINES"),

	/**
	 * The console width environment variable on *ix like shells.
	 */
	TERMINAL_WIDTH("COLUMNS"),

	/**
	 * Stands for the chars per row to be taken by REFCODES.ORG artifacts.
	 */
	CONSOLE_WIDTH("CONSOLE_WIDTH"),

	/**
	 * Stands for the lines per column to be taken by REFCODES.ORG artifacts.
	 */
	CONSOLE_HEIGHT("CONSOLE_HEIGHT"),

	/**
	 * The console's line-break property. Used to override any default line
	 * breaks for the REFCODES-ORG artifacts.
	 */
	CONSOLE_LINE_BREAK("CONSOLE_LINE_BREAK"),

	/**
	 * Set to "true" or "false", forces ANSI to be used / not used by
	 * REFCODES.ORG artifacts, no matter what capabilities were detected for the
	 * hosting terminal.
	 */
	CONSOLE_ANSI("CONSOLE_ANSI"),

	/**
	 * Stands for the chosen layout for the REFCODES.ORG logger artifacts.
	 */
	LOGGER_LAYOUT("LOGGER_LAYOUT"),

	/**
	 * Stands for the chosen style for the REFCODES.ORG logger artifacts.
	 */
	LOGGER_STYLE("LOGGER_STYLE"),

	/**
	 * The environment variable (on windows) holding the computer's name.
	 */
	COMPUTERNAME("COMPUTERNAME"),

	/**
	 * Environment variable holding the user's home folder path.
	 */
	USER_HOME("HOME"),

	/**
	 * Environment variable holding the current session's terminal.
	 */
	TERM("TERM"),

	HTTP_PROXY("HTTP_PROXY"),

	HTTPS_PROXY("HTTPS_PROXY"),

	NO_PROXY("NO_PROXY");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _environmentPropertyName;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new environment property.
	 *
	 * @param aEnvironmentPropertyName the a environment property name
	 */
	private EnvironmentProperty( String aEnvironmentPropertyName ) {
		_environmentPropertyName = aEnvironmentPropertyName;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getKey() {
		return _environmentPropertyName;
	}

	/**
	 * Retrieves the value, first the upper-case variant of the environment
	 * variable is tried out. If not set or empty, then the lower-case variant
	 * is used.
	 * 
	 * @return The value of the environment variable.
	 */
	@Override
	public String getValue() {
		String theValue = System.getenv( _environmentPropertyName.toUpperCase() );
		if ( theValue == null || theValue.length() == 0 ) {
			theValue = System.getenv( _environmentPropertyName.toLowerCase() );
		}
		return theValue;
	}

	/**
	 * Retrieves the value, first the upper-case variant of the environment
	 * variable is tried out. If not set or empty, then the lower-case variant
	 * is used.
	 *
	 * @param aValue the new value
	 */
	@Override
	public void setValue( String aValue ) {
		String theValue = System.getenv( _environmentPropertyName.toUpperCase() );
		if ( theValue == null || theValue.length() == 0 ) {
			theValue = System.getenv( _environmentPropertyName.toLowerCase() );
			if ( theValue != null && theValue.length() != 0 ) {
				System.getenv().put( _environmentPropertyName.toLowerCase(), aValue );
				return;
			}
		}
		System.getenv().put( _environmentPropertyName.toUpperCase(), aValue );
	}

}