// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.data;

import org.refcodes.mixin.NameAccessor;

/**
 * The {@link Protocol} define values useful when working with files or a
 * file-system.
 */
public enum Protocol implements NameAccessor {

	JAR("jar", FileNameExtension.JAR.getExtension() + Delimiter.JAR_URL_RESOURCE.getChar()),

	ZIP("zip", FileNameExtension.ZIP.getExtension() + Delimiter.JAR_URL_RESOURCE.getChar()),

	SH("sh", FileNameExtension.SH.getExtension() + Delimiter.JAR_URL_RESOURCE.getChar()),

	FILE("file", "file" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar()),

	HTTP("http", "http" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar() + Delimiter.PATH.getChar()),

	HTTPS("https", "https" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar() + Delimiter.PATH.getChar()),

	SOCKS("socks", "socks" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar() + Delimiter.PATH.getChar()),

	SOCKS4("socks4", "socks4" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar() + Delimiter.PATH.getChar()),

	SOCKS5("socks4", "socks4" + Suffix.PROTOCOL.getSuffix() + Delimiter.PATH.getChar() + Delimiter.PATH.getChar());

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _name;

	private String _resourceLocatorPrefix;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new protocol.
	 *
	 * @param aProtocolName the a protocol name
	 * @param aResourceLocatorPrefix the a resource locator prefix
	 */
	private Protocol( String aProtocolName, String aResourceLocatorPrefix ) {
		_name = aProtocolName;
		_resourceLocatorPrefix = aResourceLocatorPrefix;

	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return _name;
	}

	/**
	 * Returns the path prefix for the protocol being the protocol name and the
	 * protocol suffix (":"), e.g. "jar:" or "file:".
	 * 
	 * @return The protocol prefix.
	 */
	public String toPrefix() {
		return _name + Suffix.PROTOCOL.getSuffix();
	}

	/**
	 * Returns the locator part pointing into the resource of the given URL.
	 * 
	 * @param aUrl The URL from which to retrieve the locator part.
	 * @param aBeginIndex The index from where to start.
	 * 
	 * @return The locator part of the given URL or null if there is no valid
	 *         resource protocol found.
	 */
	public String toResourceLocator( String aUrl, int aBeginIndex ) {
		String theLocator = aUrl;
		int i = theLocator.indexOf( _resourceLocatorPrefix, aBeginIndex );
		if ( i != -1 ) {
			i += (_resourceLocatorPrefix.length() - 1);
			return theLocator.substring( 0, i );
		}
		return null;
	}

	/**
	 * Returns the locator part pointing into the resource of the given URL.
	 * 
	 * @param aUrl The URL from which to retrieve the locator part.
	 * 
	 * @return The locator part of the given URL or null if there is no valid
	 *         resource protocol found.
	 */
	public String toResourceLocator( String aUrl ) {
		return toResourceLocator( aUrl, 0 );
	}

	/**
	 * Returns the index of the locator part pointing into the resource of the
	 * given URL.
	 * 
	 * @param aUrl The URL from which to retrieve the locator part.
	 * @param aBeginIndex The index from where to start.
	 * 
	 * @return The index of locator part of the given URL or -1 if there is no
	 *         valid resource protocol found.
	 */
	public int nextResourceLocatorIndex( String aUrl, int aBeginIndex ) {
		int i = aUrl.indexOf( _resourceLocatorPrefix, aBeginIndex );
		if ( i != -1 ) {
			i += (_resourceLocatorPrefix.length() - 1);
		}
		return i;
	}

	/**
	 * Returns the index of locator part pointing into the resource of the given
	 * URL.
	 * 
	 * @param aUrl The URL from which to retrieve the locator part.
	 * 
	 * @return The the index locator part of the given URL or -1 if there is no
	 *         valid resource protocol found.
	 */
	public int firstResourceLocatorIndex( String aUrl ) {
		int i = aUrl.indexOf( _resourceLocatorPrefix, 0 );
		if ( i != -1 ) {
			i += (_resourceLocatorPrefix.length() - 1);
		}
		return i;
	}

	/**
	 * Returns the prefix for the resource locator as of the {@link Protocol}.
	 * E.g. a HTTP protocol will provide the resource locator prefix "http://"
	 * and a FILE protocol will provide the resource locator prefix "file:/"
	 * prefix.
	 * 
	 * @return The resource locator's prefix.
	 */
	public String getResourceLocatorPrefix() {
		return _resourceLocatorPrefix;
	}

	/**
	 * Returns that {@link Protocol} represented by the given name.
	 * 
	 * @param aName The name for which to determine the {@link Protocol}.
	 * 
	 * @return The determined {@link Protocol} or null if none was determinable.
	 */
	public static Protocol fromName( String aName ) {
		for ( Protocol eElement : values() ) {
			if ( eElement.getName().equalsIgnoreCase( aName ) ) {
				return eElement;
			}
		}
		return null;
	}
}
