// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.data;

/**
 * The Enum Proxy.
 */
public enum Proxy {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// @formatter:off
	HTTP(EnvironmentProperty.HTTP_PROXY, SystemProperty.HTTP_PROXY_HOST, SystemProperty.HTTP_PROXY_PORT, Protocol.HTTP ),
	
	HTTPS(EnvironmentProperty.HTTPS_PROXY, SystemProperty.HTTPS_PROXY_HOST, SystemProperty.HTTPS_PROXY_PORT, Protocol.HTTPS),
	
	SOCKS(EnvironmentProperty.HTTP_PROXY, SystemProperty.SOCKS_PROXY_HOST, SystemProperty.SOCKS_PROXY_PORT, Protocol.SOCKS ),
	
	SOCKS4(EnvironmentProperty.HTTP_PROXY, SystemProperty.SOCKS_PROXY_HOST, SystemProperty.SOCKS_PROXY_PORT, Protocol.SOCKS4),
	
	SOCKS5(EnvironmentProperty.HTTP_PROXY, SystemProperty.SOCKS_PROXY_HOST, SystemProperty.SOCKS_PROXY_PORT, Protocol.SOCKS5);
	// @formatter:on

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private EnvironmentProperty _envProxy;

	private SystemProperty _sysHost;

	private SystemProperty _sysPort;

	private Protocol _protocol;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new proxy.
	 *
	 * @param aEnvProxy the a env proxy
	 * @param aSysHost the a sys host
	 * @param aSysPort the a sys port
	 * @param aProtocol the a protocol
	 */
	private Proxy( EnvironmentProperty aEnvProxy, SystemProperty aSysHost, SystemProperty aSysPort, Protocol aProtocol ) {
		_envProxy = aEnvProxy;
		_protocol = aProtocol;
		_sysHost = aSysHost;
		_sysPort = aSysPort;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Initializes the proxy from the settings of the environment variables if
	 * non according proxy settings are found in the system properties.
	 */
	public void initialize() {
		toProxySettings( _envProxy, _sysHost, _sysPort, _protocol );
	}

	/**
	 * Initializes the proxy for all proxy enumeration elements, e.g. calls
	 * {@link #initialize()} on each element in this enumeration.
	 */
	public static void initializeAll() {
		for ( Proxy eProxy : values() ) {
			eProxy.initialize();
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To proxy settings.
	 *
	 * @param aFromEnvProxy the a from env proxy
	 * @param aToSysHost the a to sys host
	 * @param aToSysPort the a to sys port
	 * @param aProtocol the a protocol
	 */
	protected static void toProxySettings( EnvironmentProperty aFromEnvProxy, SystemProperty aToSysHost, SystemProperty aToSysPort, Protocol aProtocol ) {
		if ( aToSysHost.getValue() == null || aToSysHost.getValue().length() == 0 ) {
			String theHttpProxy = aFromEnvProxy.getValue();
			if ( theHttpProxy != null ) {
				if ( theHttpProxy.toLowerCase().startsWith( aProtocol.getResourceLocatorPrefix().toLowerCase() ) ) {
					int index = theHttpProxy.lastIndexOf( Delimiter.PORT.getChar() );
					if ( index != -1 ) {
						try {
							int ePort = Integer.valueOf( theHttpProxy.substring( index + 1 ) );
							String eHost = theHttpProxy.substring( 0, index );
							aToSysHost.setValue( eHost );
							aToSysPort.setValue( ePort + "" );

						}
						catch ( Exception e ) {
							aToSysHost.setValue( theHttpProxy );
						}
					}
					else {
						aToSysHost.setValue( theHttpProxy );
					}
				}
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
