// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

/**
 * Provides an accessor for a {@link Claim} instance property.
 */
public interface ClaimAccessor {

	/**
	 * Retrieves the {@link Claim} instance from the {@link Claim} instance
	 * property.
	 * 
	 * @return The {@link Claim} instance stored by the {@link Claim} instance
	 *         property.
	 */
	Claim getClaim();

	/**
	 * Provides a mutator for a {@link Claim} instance property.
	 */
	public interface ClaimMutator {

		/**
		 * Sets the {@link Claim} instance for the {@link Claim} instance
		 * property.
		 * 
		 * @param aClaim The {@link Claim} instance to be stored by the
		 *        {@link Claim} instance property.
		 */
		void setClaim( Claim aClaim );
	}

	/**
	 * Provides a builder method for a {@link Claim} instance property returning
	 * the builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface ClaimBuilder<B extends ClaimBuilder<B>> {

		/**
		 * Sets the {@link Claim} instance for the {@link Claim} instance
		 * property.
		 * 
		 * @param aClaim The {@link Claim} instance to be stored by the
		 *        {@link Claim} instance property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withClaim( Claim aClaim );
	}

	/**
	 * Provides a {@link Claim} instance property.
	 */
	public interface ClaimProperty extends ClaimAccessor, ClaimMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link Claim} (setter)
		 * as of {@link #setClaim(Claim)} and returns the very same value
		 * (getter).
		 * 
		 * @param aClaim The {@link Claim} to set (via
		 *        {@link #setClaim(Claim)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default Claim letClaim( Claim aClaim ) {
			setClaim( aClaim );
			return aClaim;
		}
	}
}
