// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.refcodes.schema.Schemable;

/**
 * The {@link Context} describes the components and modules wired together by
 * the {@link Reactor}.
 */
@SuppressWarnings("rawtypes")
public class Context implements Schemable, DependenciesAccessor {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Dependency<?>[] _dependencies;
	private final Object[] _profiles;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Empty constructor to be used in conjunction with the
	 * {@link #initialize(Dependency[])} method!.
	 *
	 * @param aProfiles The profiles which have been applied when creating this
	 *        {@link Context}.
	 */
	protected Context( Object[] aProfiles ) {
		_profiles = aProfiles;
	}

	/**
	 * Creates the {@link Context} with the managed {@link Dependency}
	 * instances.
	 * 
	 * @param aDependencies The {@link Dependency} declarations of which the
	 *        {@link Context} consists.
	 * @param aProfiles The profiles which have been applied when creating this
	 *        {@link Context}.
	 */
	public Context( Dependency<?>[] aDependencies, Object[] aProfiles ) {
		_dependencies = aDependencies;
		_profiles = aProfiles;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Retrieves the The profiles which have been applied when creating this
	 * {@link Context}.
	 * 
	 * @return The according profiles.
	 */
	public Object[] getProfiles() {
		return _profiles;
	}

	/**
	 * Retrieves the {@link Dependency} declarations contained in the
	 * {@link Context}.
	 * 
	 * @return The {@link Dependency} declarations of which the {@link Context}
	 *         consists.
	 */
	@Override
	public Dependency<?>[] getDependencies() {
		return _dependencies;
	}

	/**
	 * Determines the {@link Dependency} element for the given instance created
	 * from the {@link Dependency}.
	 * 
	 * @param aInstance The instance which's {@link Dependency} element creating
	 *        this instance is to be retrieved.
	 * 
	 * @return The according {@link Dependency} element or null if none
	 *         {@link Dependency} element is responsible for crating the
	 *         provided instance.
	 */
	public Dependency<?> getDependencyByInstance( Object aInstance ) {
		for ( Dependency<?> eDepenendency : _dependencies ) {
			if ( eDepenendency.hasInstance( aInstance ) ) {
				return eDepenendency;
			}
		}
		return null;
	}

	/**
	 * Retrieves the {@link Dependency} declaration with the given alias managed
	 * by this {@link Context}, derived from the {@link Dependency}
	 * {@link Dependency} declaration from which the {@link Reactor} created
	 * this {@link Context}.
	 *
	 * @param aAlias The alias of {@link Dependency} declaration to retrieve.
	 * 
	 * @return The according {@link Dependency} declaration managed by this
	 *         {@link Context}.
	 */
	public Dependency<?> getDependencyByAlias( String aAlias ) {
		if ( aAlias != null ) {
			for ( Dependency<?> eDependency : _dependencies ) {
				if ( aAlias.equals( eDependency.getAlias() ) ) {
					return eDependency;
				}
			}
		}
		return null;
	}

	/**
	 * Retrieves the {@link Dependency} declarations of the given type managed
	 * by this {@link Context}, derived from the {@link Dependency} declarations
	 * from which the {@link Reactor} created this {@link Context}.
	 *
	 * @param <T> the generic type
	 * @param aType The type of {@link Dependency} declarations to retrieve.
	 * 
	 * @return The according {@link Dependency} declarations managed by this
	 *         {@link Context}.
	 */
	@SuppressWarnings("unchecked")
	public <T> Dependency<T>[] getDependenciesByType( Class<T> aType ) {
		final List<Dependency<T>> theDependnecies = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			if ( aType.isAssignableFrom( eDependency.getType() ) ) {
				theDependnecies.add( (Dependency<T>) eDependency );
			}
		}
		return theDependnecies.toArray( (Dependency<T>[]) Array.newInstance( Dependency.class, theDependnecies.size() ) );
	}

	/**
	 * Retrieves the {@link Dependency} declarations of given tags managed by
	 * this {@link Context}, derived from the {@link Dependency}
	 * {@link Dependency} declarations from which the {@link Reactor} created
	 * this {@link Context}.
	 * 
	 * @param aTags The tags of the {@link Dependency} declarations to retrieve.
	 * 
	 * @return The according {@link Dependency} declarations managed by this
	 *         {@link Context}.
	 */
	public Dependency<?>[] getDependenciesByTags( Object... aTags ) {
		final List<Dependency<?>> theDependencies = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			out: {
				for ( Object eTag : aTags ) {
					for ( Object eDependencyTag : eDependency.getTags() ) {
						if ( eTag.toString().equalsIgnoreCase( eDependencyTag.toString() ) ) {
							theDependencies.add( eDependency );
							break out;
						}
					}
				}
			}
		}
		return theDependencies.toArray( new Dependency<?>[theDependencies.size()] );
	}

	/**
	 * Retrieves the {@link Dependency} declarations of given profiles managed
	 * by this {@link Context}, derived from the {@link Dependency}
	 * {@link Dependency} declarations from which the {@link Reactor} created
	 * this {@link Context}.
	 * 
	 * @param aProfiles The profiles of the {@link Dependency} declarations to
	 *        retrieve.
	 * 
	 * @return The according {@link Dependency} declarations managed by this
	 *         {@link Context}.
	 */
	public Dependency<?>[] getDependenciesByProfiles( Object... aProfiles ) {
		final List<Dependency<?>> theDependencies = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			out: {
				for ( Object eProfile : aProfiles ) {
					for ( Object eDependencyProfile : eDependency.getProfiles() ) {
						if ( eProfile.toString().equalsIgnoreCase( eDependencyProfile.toString() ) ) {
							theDependencies.add( eDependency );
							break out;
						}
					}
				}
			}
		}
		return theDependencies.toArray( new Dependency<?>[theDependencies.size()] );
	}

	/**
	 * Retrieves the instances managed by this {@link Context}, derived from the
	 * {@link Dependency} declarations from which the {@link Reactor} created
	 * this {@link Context}.
	 * 
	 * @return The instances managed by this {@link Context}.
	 */
	public Object[] getInstances() {
		final List<Object> theInstances = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			theInstances.addAll( Arrays.asList( eDependency.getInstances() ) );
		}
		return theInstances.toArray();
	}

	/**
	 * Retrieves the instances created by a {@link Dependency} having
	 * {@link InstanceMetrics} with {@link InstanceMetrics#isSingleton()} being
	 * true (e.g. {@link InstanceMode#SINGLETON_IS_MANDATORY} or
	 * {@link InstanceMode#SINGLETON_ON_DEMAND}) and with the given alias
	 * managed by this {@link Context}, derived from the {@link Dependency}
	 * instances from which the {@link Reactor} created this {@link Context}.
	 *
	 * @param aAlias The alias of instances to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	public Object getSingletonByAlias( String aAlias ) {
		if ( aAlias != null ) {
			for ( Dependency<?> eDependency : _dependencies ) {
				if ( aAlias.equals( eDependency.getAlias() ) && eDependency.getInstanceMetrics().isSingleton() ) {
					return eDependency.getInstances() != null && eDependency.getInstances().length != 0 ? eDependency.getInstances()[0] : null;
				}
			}
		}
		return null;
	}

	/**
	 * Retrieves the instances created by a {@link Dependency} having
	 * {@link InstanceMetrics} with {@link InstanceMetrics#isSingleton()} being
	 * true (e.g. {@link InstanceMode#SINGLETON_IS_MANDATORY} or
	 * {@link InstanceMode#SINGLETON_ON_DEMAND}) and with the given alias
	 * managed by this {@link Context}, derived from the {@link Dependency}
	 * instances from which the {@link Reactor} created this {@link Context}.
	 * 
	 * @param <T> The type of the instance to be retrieved.
	 *
	 * @param aAlias The alias of instances to retrieve.
	 * @param aType The type of the instance to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	public <T> T getSingletonByAlias( String aAlias, Class<T> aType ) {
		if ( aAlias != null ) {
			for ( Dependency<T> eDependency : getDependenciesByType( aType ) ) {
				if ( aAlias.equals( eDependency.getAlias() ) && eDependency.getInstanceMetrics().isSingleton() ) {
					return eDependency.getInstances() != null && eDependency.getInstances().length != 0 ? eDependency.getInstances()[0] : null;
				}
			}
		}
		return null;
	}

	/**
	 * Retrieves the instances with the given alias managed by this
	 * {@link Context}, derived from the {@link Dependency} declarations from
	 * which the {@link Reactor} created this {@link Context}.
	 *
	 * @param aAlias The alias of instances to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	public Object[] getInstancesByAlias( String aAlias ) {
		if ( aAlias != null ) {
			for ( Dependency<?> eDependency : _dependencies ) {
				if ( aAlias.equals( eDependency.getAlias() ) ) {
					return eDependency.getInstances();
				}
			}
		}
		return null;
	}

	/**
	 * Retrieves the instances of the given type managed by this
	 * {@link Context}, derived from the {@link Dependency} declarations from
	 * which the {@link Reactor} created this {@link Context}.
	 *
	 * @param <T> the generic type
	 * @param aType The type of instances to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	@SuppressWarnings("unchecked")
	public <T> T[] getInstancesByType( Class<T> aType ) {
		final List<Object> theInstances = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			if ( aType.isAssignableFrom( eDependency.getType() ) ) {
				theInstances.addAll( Arrays.asList( eDependency.getInstances() ) );
			}
		}
		return theInstances.toArray( (T[]) Array.newInstance( aType, theInstances.size() ) );
	}

	/**
	 * Retrieves the first instance of the given type managed by this
	 * {@link Context}, derived from the {@link Dependency} declarations from
	 * which the {@link Reactor} created this {@link Context}.
	 *
	 * @param <T> the generic type
	 * @param aType The type of instance to retrieve.
	 * 
	 * @return The according first instance managed by this {@link Context} or
	 *         null if there is none.
	 */
	public <T> T getFirstByType( Class<T> aType ) {
		final T[] theInstances = getInstancesByType( aType );
		return theInstances.length > 0 ? theInstances[0] : null;
	}

	/**
	 * Retrieves the instances of given tags managed by this {@link Context},
	 * derived from the {@link Dependency} declarations from which the
	 * {@link Reactor} created this {@link Context}.
	 * 
	 * @param aTags The tags of the instances to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	public Object[] getInstancesByTags( Object... aTags ) {
		final List<Object> theInstances = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			out: {
				for ( Object eTag : aTags ) {
					for ( Object eDependencyTag : eDependency.getTags() ) {
						if ( eTag.toString().equalsIgnoreCase( eDependencyTag.toString() ) ) {
							theInstances.addAll( Arrays.asList( eDependency.getInstances() ) );
							break out;
						}
					}
				}
			}
		}
		return theInstances.toArray();
	}

	/**
	 * Retrieves the instances of given profiles managed by this
	 * {@link Context}, derived from the {@link Dependency} declarations from
	 * which the {@link Reactor} created this {@link Context}.
	 * 
	 * @param aProfiles The profiles of the instances to retrieve.
	 * 
	 * @return The according instances managed by this {@link Context}.
	 */
	public Object[] getInstancesByProfiles( Object... aProfiles ) {
		final List<Object> theInstances = new ArrayList<>();
		for ( Dependency<?> eDependency : _dependencies ) {
			out: {
				for ( Object eProfile : aProfiles ) {
					for ( Object eDependencyProfile : eDependency.getProfiles() ) {
						if ( eProfile.toString().equalsIgnoreCase( eDependencyProfile.toString() ) ) {
							theInstances.addAll( Arrays.asList( eDependency.getInstances() ) );
							break out;
						}
					}
				}
			}
		}
		return theInstances.toArray();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DependencySchema toSchema() {
		DependencySchema[] theSchemas = null;
		if ( _dependencies != null && _dependencies.length != 0 ) {
			final Set<Dependency<?>> theDependnecies = new HashSet<>();
			final Set<DependencySchema> theSet = new LinkedHashSet<>();
			for ( Dependency<?> eDependency : _dependencies ) {
				if ( eDependency.getInstanceMetrics() != null && eDependency.getInstanceMetrics().isMandatory() || !eDependency.hasInstances() ) {
					if ( theDependnecies.add( eDependency ) ) {
						theSet.add( eDependency.toSchema() );
					}
				}
				if ( eDependency.getFactory() != null && eDependency.getFactory().getAlias() != null ) {
					final Dependency<?> eFactory = getDependencyByAlias( eDependency.getFactory().getAlias() );
					if ( eFactory != null ) {
						if ( theDependnecies.add( eFactory ) ) {
							theSet.add( eFactory.toSchema() );
						}
					}
				}
			}
			theSchemas = theSet.toArray( new DependencySchema[theSet.size()] );
		}
		return new DependencySchema( getClass(), "Context managing dependencies between modules.", theSchemas );
	}

	// /////////////////////////////////////////////////////////////////////////
	// MODULE HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Initializes the {@link Context} after construction so it can already be
	 * provided to {@link Dependency} declarations.
	 *
	 * @param aDependencies the dependencies
	 */
	void initialize( Dependency<?>[] aDependencies ) {
		_dependencies = aDependencies;
	}
}
