// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

/**
 * Provides an accessor for a dependency property.
 * 
 * @param <T> The generic type of the dependency.
 */
public interface DependencyAccessor<T> {

	/**
	 * Retrieves the dependency from the dependency property.
	 * 
	 * @return The dependency stored by the dependency property.
	 */
	Dependency<T> getDependency();

	/**
	 * Provides a mutator for a dependency property.
	 * 
	 * @param <T> The generic type of the dependency.
	 */
	public interface DependencyMutator<T> {

		/**
		 * Sets the dependency for the dependency property.
		 * 
		 * @param aDependency The dependency to be stored by the dependency
		 *        property.
		 */
		void setDependency( Dependency<T> aDependency );
	}

	/**
	 * Provides a builder method for a dependency property returning the builder
	 * for applying multiple build operations.
	 * 
	 * @param <T> The generic type of the dependency.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface DependencyBuilder<T, B extends DependencyBuilder<T, B>> {

		/**
		 * Sets the dependency for the dependency property.
		 * 
		 * @param aDependency The dependency to be stored by the dependency
		 *        property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withDependency( Dependency<T> aDependency );
	}

	/**
	 * Provides a dependency property.
	 * 
	 * @param <T> The generic type of the dependency.
	 */
	public interface DependencyProperty<T> extends DependencyAccessor<T>, DependencyMutator<T> {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link Dependency<T>}
		 * (setter) as of {@link #setDependency(Dependency)} and returns the
		 * very same value (getter).
		 * 
		 * @param aDependency The {@link Dependency<T>} to set (via
		 *        {@link #setDependency(Dependency)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default Dependency<T> letDependency( Dependency<T> aDependency ) {
			setDependency( aDependency );
			return aDependency;
		}
	}
}
