// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import org.refcodes.mixin.CommentAccessor;
import org.refcodes.mixin.SizeAccessor;
import org.refcodes.schema.Schema;

/**
 * The purpose of a {@link DependencySchema} is to automatically outline and
 * generate documentation of {@link Dependency} configuration.
 */
public class DependencySchema extends Schema implements CommentAccessor, SizeAccessor {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	public static final String DEPENDENCY = "DEPENDENCY";
	public static final String FACTORY = "FACTORY";
	public static final String INSTANCES = "INSTANCES";
	public static final String MANDATORY = "MANDATORY";
	public static final String PROFILES = "PROFILES";
	public static final String SETUP = "SETUP";
	public static final String SINGLETON = "SINGLETON";
	public static final String SIZE = "SIZE";
	public static final String TAGS = "TAGS";

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	public DependencySchema( Class<?> aType, String aDescription, Schema... aChildren ) {
		super( aType, aDescription, aChildren );
	}

	/**
	 * {@inheritDoc}
	 */
	public DependencySchema( String aAlias, Class<?> aType, String aDescription ) {
		super( aAlias, aType, aDescription );
	}

	/**
	 * {@inheritDoc}
	 */
	public DependencySchema( String aAlias, Class<?> aType, String aDescription, Schema... aChildren ) {
		super( aAlias, aType, aDescription, aChildren );
	}

	/**
	 * Constructs a {@link DependencySchema} from the provided {@link Claim}.
	 * 
	 * @param aClaim The {@link Claim} from which to construct the schema.
	 */
	public DependencySchema( Claim aClaim ) {
		super( aClaim.getAlias(), aClaim.getClass(), "A dependency claimed to be satisfied." );
		put( DEPENDENCY, aClaim.getType() );
		put( COMMENT, aClaim.toString() );
	}

	/**
	 * Constructs a {@link DependencySchema} from the provided
	 * {@link FactoryClaim}.
	 * 
	 * @param aFactoryClaim The {@link FactoryClaim} from which to construct the
	 *        schema.
	 */
	public DependencySchema( FactoryClaim<?, ?> aFactoryClaim ) {
		super( aFactoryClaim.getAlias(), aFactoryClaim.getClass(), "A factory being claimed to be applied by a dependency when creating its instances." );
		put( FACTORY, aFactoryClaim.getFactory().getClass() );
		put( DEPENDENCY, aFactoryClaim.getType() );
		put( COMMENT, aFactoryClaim.toString() );
	}

	/**
	 * Constructs a {@link DependencySchema} from the provided
	 * {@link InitializerClaim}.
	 * 
	 * @param aInitializerClaim The {@link InitializerClaim} from which to
	 *        construct the schema.
	 */
	public DependencySchema( InitializerClaim<?, ?> aInitializerClaim ) {
		super( aInitializerClaim.getAlias(), aInitializerClaim.getClass(), "An initializer being claimed to be applied by a dependency when setting up its instances." );
		put( SETUP, aInitializerClaim.getSetup().getClass() );
		put( DEPENDENCY, aInitializerClaim.getType() );
		put( COMMENT, aInitializerClaim.toString() );
	}

	/**
	 * Constructs a {@link DependencySchema} from the provided
	 * {@link Dependency} and the provided arguments.
	 *
	 * @param aDependency The {@link Dependency} from which to retrieve the meta
	 *        data of the schema.
	 * @param aChildren The children's {@link DependencySchema} descriptions
	 *        representing part of the {@link DependencySchema} providing type.
	 */
	public DependencySchema( Dependency<?> aDependency, Schema... aChildren ) {
		super( aDependency.getAlias(), aDependency.getClass(), toDescription( aDependency ), aChildren );
		if ( aDependency.getInstanceMetrics() != null ) {
			put( MANDATORY, aDependency.getInstanceMetrics().isMandatory() );
			put( SINGLETON, aDependency.getInstanceMetrics().isSingleton() );
		}
		if ( aDependency.getTags() != null && aDependency.getTags().length != 0 ) {
			final String[] theTags = new String[aDependency.getTags().length];
			for ( int i = 0; i < aDependency.getTags().length; i++ ) {
				theTags[i] = aDependency.getTags()[i] != null ? aDependency.getTags()[i].toString() : null;
			}
			put( TAGS, theTags );
		}
		if ( aDependency.getProfiles() != null && aDependency.getProfiles().length != 0 ) {
			final String[] theProfiles = new String[aDependency.getProfiles().length];
			for ( int i = 0; i < aDependency.getProfiles().length; i++ ) {
				theProfiles[i] = aDependency.getProfiles()[i] != null ? aDependency.getProfiles()[i].toString() : null;
			}
			put( PROFILES, theProfiles );
		}
		if ( aDependency.getInstances() != null && aDependency.getInstances().length != 0 ) {
			final String[] theInstances = new String[aDependency.getInstances().length];
			for ( int i = 0; i < aDependency.getInstances().length; i++ ) {
				theInstances[i] = aDependency.getInstances()[i] != null ? aDependency.getInstances()[i].toString() : null;
			}
			put( INSTANCES, theInstances );
		}
		if ( aDependency.getFactory() != null ) {
			put( FACTORY, aDependency.getFactory() );
		}
		if ( aDependency.getSetup() != null ) {
			put( SETUP, aDependency.getSetup() );
		}
		put( SIZE, aDependency.getInstances().length );
		put( DEPENDENCY, aDependency.getType() );
		put( COMMENT, aDependency.toString() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Retrieves the verbose representation of the value.
	 * 
	 * @return The human readable value representation.
	 */
	@Override
	public String getComment() {
		return (String) get( COMMENT );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSize() {
		final Integer theSize = (Integer) get( SIZE );
		return theSize != null ? theSize : -1;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static String toDescription( Dependency<?> aDependency ) {
		String theDescription = "";
		if ( aDependency.getInstanceMetrics() != null ) {
			if ( aDependency.getInstanceMetrics().isSingleton() ) {
				theDescription += "Singleton";
			}
			else {
				theDescription += "Non-Singleton";
			}
		}
		else {
			theDescription += "Undefined";
		}
		theDescription += " dependency";
		if ( aDependency.getType() != null ) {
			theDescription += " of type <" + aDependency.getType().getSimpleName() + ">";
		}
		else {
			theDescription += " of unspecified type";
		}
		return theDescription;
	}
}