// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import java.util.function.Function;

import org.refcodes.factory.ContextTypeFactory;

/**
 * A {@link FactoryClaim} extends a {@link Claim} and uses a {@link Function}
 * upon a {@link Dependency} to create (fabricate) an instance.
 *
 * @param <C> The type of the {@link Dependency} claimed for fabricating
 *        instances of the given type.
 * @param <T> The type of the instance to be created.
 */
public class FactoryClaim<C, T> extends Claim implements ContextTypeFactory<T, C> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private final Function<C, T> _factory;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a new {@link FactoryClaim}.
	 * 
	 * @param aType The type of the {@link Dependency} required by the
	 *        {@link FactoryClaim}.
	 * @param aFactory The processor to be used by the {@link FactoryClaim}
	 *        instance.
	 */
	public FactoryClaim( Class<C> aType, Function<C, T> aFactory ) {
		super( aType );
		_factory = aFactory;
	}

	/**
	 * Creates a new {@link FactoryClaim}.
	 *
	 * @param aType The type of the {@link Dependency} required by the
	 *        {@link FactoryClaim}.
	 * @param aFactory The processor to be used by the {@link FactoryClaim}
	 *        instance.
	 * @param aAlias The alias for the {@link FactoryClaim}.
	 */
	public FactoryClaim( Class<C> aType, Function<C, T> aFactory, String aAlias ) {
		super( aType, aAlias );
		_factory = aFactory;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Fabricates (creates) the provided instance.
	 * 
	 * @param aClaim The {@link Dependency} declaration required for
	 *        fabrication.
	 * 
	 * @return The fabricated instance.
	 */
	@Override
	public T create( C aClaim ) {
		return _factory.apply( aClaim );
	}

	/**
	 * Returns the type of the {@link Dependency} required by this
	 * {@link FactoryClaim} instance when invoking {@link #create(Object)}.
	 * 
	 * @return The according type.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Class<C> getType() {
		return (Class<C>) _type;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DependencySchema toSchema() {
		return new DependencySchema( this );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the factory.
	 *
	 * @return the factory
	 */
	Function<C, T> getFactory() {
		return _factory;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Forces calling {@link #initialize(Object, Object)} by doing an unsafe
	 * cast.
	 *
	 * @param aClaim The {@link Dependency} declaration required for
	 *        initialization.
	 * 
	 * @return The initialized instance.
	 */
	@SuppressWarnings("unchecked")
	T toInstance( Object aClaim ) {
		return create( (C) aClaim );
	}
}