// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import java.util.function.BiFunction;

/**
 * A {@link InitializerClaim} extends a {@link Claim} and uses a
 * {@link BiFunction} upon a {@link Dependency} to initialize a given instance.
 *
 * @param <C> The type of the {@link Dependency} claimed for setting up an
 *        instance of the given type.
 * @param <T> The type of the instance to be initialize.
 */
public class InitializerClaim<C, T> extends Claim {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private final BiFunction<T, C, T> _initializer;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a new {@link InitializerClaim}.
	 * 
	 * @param aType The type of the {@link Dependency} required by the
	 *        {@link InitializerClaim}.
	 * @param aInitializer The processor to be used by the
	 *        {@link InitializerClaim} instance.
	 */
	public InitializerClaim( Class<C> aType, BiFunction<T, C, T> aInitializer ) {
		super( aType );
		_initializer = aInitializer;
	}

	/**
	 * Creates a new {@link InitializerClaim}.
	 *
	 * @param aType The type of the {@link Dependency} required by the
	 *        {@link InitializerClaim}.
	 * @param aInitializer The processor to be used by the
	 *        {@link InitializerClaim} instance.
	 * @param aAlias The alias for the {@link InitializerClaim}.
	 */
	public InitializerClaim( Class<C> aType, BiFunction<T, C, T> aInitializer, String aAlias ) {
		super( aType, aAlias );
		_initializer = aInitializer;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Initializes the provided instance.
	 * 
	 * @param aClaim The {@link Dependency} declaration required for
	 *        initialization.
	 * @param aInstance The instance to be initialized.
	 * 
	 * @return The initialized instance.
	 */
	public T initialize( C aClaim, T aInstance ) {
		return _initializer.apply( aInstance, aClaim );
	}

	/**
	 * Returns the type of the {@link Dependency} required by this
	 * {@link InitializerClaim} instance when invoking
	 * {@link #initialize(Object, Object)}.
	 * 
	 * @return The according type.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Class<C> getType() {
		return (Class<C>) _type;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DependencySchema toSchema() {
		return new DependencySchema( this );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the setup.
	 *
	 * @return the setup
	 */
	BiFunction<T, C, T> getSetup() {
		return _initializer;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Forces calling {@link #initialize(Object, Object)} by doing an unsafe
	 * cast.
	 *
	 * @param aClaim The {@link Dependency} declaration required for
	 *        initialization.
	 * @param aInstance The instance to be initialized.
	 * 
	 * @return The initialized instance.
	 */
	@SuppressWarnings("unchecked")
	T toInstance( Object aClaim, T aInstance ) {
		return initialize( (C) aClaim, aInstance );
	}
}
