// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

/**
 * Provides an accessor for a {@link InstanceMetrics} property.
 */
public interface InstanceMetricsAccessor {

	/**
	 * Retrieves the {@link InstanceMetrics} instance from the
	 * {@link InstanceMetrics} property.
	 * 
	 * @return The {@link InstanceMetrics} instance stored by the
	 *         {@link InstanceMetrics} property.
	 */
	InstanceMetrics getInstanceMetrics();

	/**
	 * Provides a mutator for a {@link InstanceMetrics} property.
	 */
	public interface InstanceMetricsMutator {

		/**
		 * Sets the {@link InstanceMetrics} instance for the
		 * {@link InstanceMetrics} property.
		 * 
		 * @param aInstanceMetrics The {@link InstanceMetrics} instance to be
		 *        stored by the {@link InstanceMetrics} property.
		 */
		void setInstanceMetrics( InstanceMetrics aInstanceMetrics );
	}

	/**
	 * Provides a builder method for a {@link InstanceMetrics} property
	 * returning the builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface InstanceMetricsBuilder<B extends InstanceMetricsBuilder<B>> {

		/**
		 * Sets the {@link InstanceMetrics} instance for the
		 * {@link InstanceMetrics} property.
		 * 
		 * @param aInstanceMetrics The {@link InstanceMetrics} instance to be
		 *        stored by the {@link InstanceMetrics} property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withInstanceMetrics( InstanceMetrics aInstanceMetrics );
	}

	/**
	 * Provides a {@link InstanceMetrics} property.
	 */
	public interface InstanceMetricsProperty extends InstanceMetricsAccessor, InstanceMetricsMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link InstanceMetrics}
		 * (setter) as of {@link #setInstanceMetrics(InstanceMetrics)} and
		 * returns the very same value (getter).
		 * 
		 * @param aInstanceMetrics The {@link InstanceMetrics} to set (via
		 *        {@link #setInstanceMetrics(InstanceMetrics)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default InstanceMetrics letInstanceMetrics( InstanceMetrics aInstanceMetrics ) {
			setInstanceMetrics( aInstanceMetrics );
			return aInstanceMetrics;
		}
	}
}
