// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

/**
 * Provides an accessor for a tags property.
 */
public interface TagsAccessor {

	/**
	 * Retrieves the tags from the tags property.
	 * 
	 * @return The tags stored by the tags property.
	 */
	Object[] getTags();

	/**
	 * Provides a mutator for a tags property.
	 */
	public interface TagsMutator {

		/**
		 * Sets the tags for the tags property.
		 * 
		 * @param aTags The tags to be stored by the tags property.
		 */
		void setTags( Object[] aTags );
	}

	/**
	 * Provides a builder method for a tags property returning the builder for
	 * applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface TagsBuilder<B extends TagsBuilder<B>> {

		/**
		 * Sets the tags for the tags property.
		 * 
		 * @param aTags The tags to be stored by the tags property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withTags( Object... aTags );
	}

	/**
	 * Provides a tags property.
	 */
	public interface TagsProperty extends TagsAccessor, TagsMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link Object[]}
		 * (setter) as of {@link #setTags(Object[])} and returns the very same
		 * value (getter).
		 * 
		 * @param aTags The {@link Object[]} to set (via
		 *        {@link #setTags(Object[])}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default Object[] letTags( Object[] aTags ) {
			setTags( aTags );
			return aTags;
		}
	}
}
