// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.filesystem.alt.s3;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;

import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.PropertiesCredentials;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;

//@formatter:off
/**
 * Abstract class to be used for any S3 using service. An endpoint can be
 * specified (i.e. the location where Amazon SimpleDB will store the data):
 * 
 * Possible endpoints for SimpleDB can be retrieved from here:
 *
 * See "http://docs.amazonwebservices.com/general/latest/gr/rande.html#s3_region"
 * 
 *	<table summary="List of AWS Regions and Endpoints">
 * 		<tr>
 *			<td><strong>Region</strong></td><td><strong>Endpoint</strong></td><td><strong>Location Constraint</strong></td><td><strong>Protocol</strong></td>
 *		</tr>
 *		<tr>
 *			<td>US Standard *</td><td>s3.amazonaws.com</td><td>(none required)</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>US West (Oregon)</td><td>s3-us-west-2.amazonaws.com</td><td>us-west-2</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>US West (Northern California)</td><td>s3-us-west-1.amazonaws.com</td><td>us-west-1</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>EU (Ireland)</td><td>s3-eu-west-1.amazonaws.com</td><td>EU</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>Asia Pacific (Singapore)</td><td>s3-ap-southeast-1.amazonaws.com</td><td>ap-southeast-1</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>Asia Pacific (Tokyo)</td><td>s3-ap-northeast-1.amazonaws.com</td><td>ap-northeast-1</td><td>HTTP and HTTPS</td>
 *		</tr>
 *		<tr>
 *			<td>South America (Sao Paulo)</td><td>s3-sa-east-1.amazonaws.com</td><td>sa-east-1</td><td>HTTP and HTTPS</td>
 *		</tr>
 *	</table>
 */
//@formatter:on
abstract class AbstractS3Client {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int THREAD_POOL_SIZE = 20;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private static final String DEFAULT_REGION = "s3-eu-west-1.amazonaws.com";

	private AmazonS3 _amazonS3Client = null;

	private String _amazonS3BucketName = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the S3 support by directly providing all needed information to
	 * setup the instance.
	 * 
	 * @param aBucketName The name of the bucket to use.
	 * @param aAccessKey The access key to use.
	 * @param aSecretKey The secret key to use.
	 */
	public AbstractS3Client( String aBucketName, String aAccessKey, String aSecretKey ) {
		this( aBucketName, aAccessKey, aSecretKey, DEFAULT_REGION );
	}

	//@formatter:off
	/**
	 * Constructs the S3 support by directly providing all needed information to
	 * setup the instance. An endpoint can be specified, i.e. the
	 * region where Amazon will store the data. 
	 * 
	 * Possible endpoints for SimpleDB can be retrieved  as stated above.
	 * 
	 * {@link http://docs.amazonwebservices.com/general/latest/gr/rande.html#s3_region}
	 * 
	 * @param aBucketName The name of the bucket to use.
	 * @param aAccessKey The access key to use.
	 * @param aSecretKey The secret key to use.
	 * @param aRegion The endpoint (Amazon region) to use.
	 */
	//@formatter:on
	public AbstractS3Client( String aBucketName, String aAccessKey, String aSecretKey, String aRegion ) {
		if ( aRegion == null ) {
			aRegion = DEFAULT_REGION;
		}
		_amazonS3Client = AmazonS3ClientBuilder.standard().withCredentials( new AWSStaticCredentialsProvider( new BasicAWSCredentials( aAccessKey, aSecretKey ) ) ).withRegion( aRegion ).build();
		_amazonS3BucketName = aBucketName;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Retrieves the bucket name to be used.
	 * 
	 * @return The bucket name.
	 */
	protected String getAmazonS3BucketName() {
		return _amazonS3BucketName;
	}

	/**
	 * Sets the bucket name to be used.
	 * 
	 * @param aAmazonS3BucketName The bucket name to be used.
	 * 
	 */
	protected void setAmazonS3BucketName( String aAmazonS3BucketName ) {
		_amazonS3BucketName = aAmazonS3BucketName;
	}

	/**
	 * Retrieves the amazon S3 client to be used.
	 * 
	 * @return The S3 client to be used.
	 */
	protected AmazonS3 getAmazonS3Client() {
		return _amazonS3Client;
	}

	/**
	 * Creates an S3 bucket.
	 * 
	 * @param aAmazonS3 The {@link AmazonS3} client.
	 * @param aBucketId The ID of the bucket to be created.
	 */
	public static void createBucket( AmazonS3 aAmazonS3, String aBucketId ) {
		aAmazonS3.createBucket( aBucketId );
	}

	/**
	 * Clears (removes all content from) an S3 bucket.
	 * 
	 * @param aAmazonS3 The {@link AmazonS3} client.
	 * 
	 * @param aBucketId The ID of the bucket to be cleared.
	 */
	protected static void clearBucket( final AmazonS3 aAmazonS3, final String aBucketId ) {
		ObjectListing theObjectListing = aAmazonS3.listObjects( aBucketId );
		while ( theObjectListing.getObjectSummaries() != null && !theObjectListing.getObjectSummaries().isEmpty() ) {
			List<S3ObjectSummary> theObjectSummaries = theObjectListing.getObjectSummaries();
			deleteS3Objects( aAmazonS3, aBucketId, theObjectSummaries );
			theObjectListing = aAmazonS3.listNextBatchOfObjects( theObjectListing );
		}
	}

	/**
	 * Deletes an S3 bucket.
	 * 
	 * @param aAmazonS3 The {@link AmazonS3} client.
	 * 
	 * @param aBucketId The ID of the bucket to be deleted.
	 */
	public static void deleteBucket( final AmazonS3 aAmazonS3, final String aBucketId ) {
		clearBucket( aAmazonS3, aBucketId );
		aAmazonS3.deleteBucket( aBucketId );
	}

	/**
	 * Creates an {@link AmazonS3} "client".
	 * 
	 * @param aAccessKey The according access key for accessing amazon AWS.
	 * @param aSecretKey The secret access key for accessing amazon AWS.
	 * 
	 * @return The client represented by an {@link AmazonS3} instance.
	 */
	protected static AmazonS3 createAmazonS3( String aAccessKey, String aSecretKey ) {
		return AmazonS3ClientBuilder.standard().withCredentials( new AWSStaticCredentialsProvider( new BasicAWSCredentials( aAccessKey, aSecretKey ) ) ).build();
	}

	/**
	 * Deletes the content described by the given {@link S3ObjectSummary} list
	 * from an S3 bucket.
	 * 
	 * @param aAmazonS3 The {@link AmazonS3} client.
	 * 
	 * @param aBucketId The ID of the bucket from which the objects are to be
	 *        deleted.
	 * 
	 * @param aS3SummaryObjects The {@link S3ObjectSummary} list describing the
	 *        objects to be deleted.
	 */
	protected static void deleteS3Objects( final AmazonS3 aAmazonS3, final String aBucketId, List<S3ObjectSummary> aS3SummaryObjects ) {
		ExecutorService theExecutorService = Executors.newFixedThreadPool( THREAD_POOL_SIZE );
		final CountDownLatch theCountDownLatch = new CountDownLatch( aS3SummaryObjects.size() );
		for ( final S3ObjectSummary eSummary : aS3SummaryObjects ) {
			theExecutorService.execute( new Runnable() {

				/**
				 * {@inheritDoc}
				 */
				@Override
				public void run() {
					try {
						aAmazonS3.deleteObject( aBucketId, eSummary.getKey() );
					}
					catch ( Exception e ) {
						LOGGER.warn( ExceptionUtility.toMessage( e ), e );
					}
					finally {
						theCountDownLatch.countDown();
					}
				}
			} );
		}
		try {
			theCountDownLatch.await();
		}
		catch ( InterruptedException ignored ) {}
		theExecutorService.shutdown();
	}

	/**
	 * Retrieves an {@link AmazonS3Client} from a configuration file containing
	 * the access- and the secret key.
	 * 
	 * @param aConfigFile The configuration file used to configure the
	 *        {@link AmazonS3Client}.
	 * 
	 * @return An {@link AmazonS3Client}.
	 * 
	 * @throws IOException In case there were problems reading the configuration
	 *         file.
	 */
	protected static AmazonS3 getAmazonS3Client( File aConfigFile ) throws IOException {
		Properties theProperties = new Properties();
		theProperties.load( new FileInputStream( aConfigFile ) );
		return AmazonS3ClientBuilder.standard().withCredentials( new AWSStaticCredentialsProvider( new PropertiesCredentials( aConfigFile ) ) ).build();
	}
}
