// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.filesystem.impls;

import java.util.Date;

import org.refcodes.filesystem.FileHandle;
import org.refcodes.filesystem.FileSystemUtility;

/**
 * Straight forward implementation of the {@link FileHandle},
 */
public class FileHandleImpl implements FileHandle {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected String _path;
	protected String _name;
	protected long _size;
	protected Date _createdDate;
	protected Date _modifiedDate;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link FileHandle} object with the given properties.
	 * 
	 * @param aPath The path to which the file handle points.
	 * @param aName The name to which the path points
	 * @param aSize The size of the file
	 * @param aCreatedDate The creation date of the file
	 * @param aModifiedDate The modified date of the file
	 */
	public FileHandleImpl( String aPath, String aName, long aSize, Date aCreatedDate, Date aModifiedDate ) {
		super();
		_path = aPath;
		_name = aName;
		_size = aSize;
		_createdDate = aCreatedDate;
		_modifiedDate = aModifiedDate;
	}

	/**
	 * Constructs a {@link FileHandle} object with the given properties.
	 * 
	 * @param aKey The path and the name (= the key) to which the file handle
	 *        points.
	 */
	public FileHandleImpl( String aKey ) {
		super();
		_path = FileSystemUtility.getPath( aKey );
		_name = FileSystemUtility.getName( aKey );
	}

	/**
	 * Constructs a {@link FileHandle} object with the given properties.
	 * 
	 * @param aPath The path to which the file handle points.
	 * @param aName The name to which the path points
	 * @param aSize The size of the file
	 * @param aCreatedDate The creation date of the file
	 * @param aModifiedDate The modified date of the file
	 */
	public FileHandleImpl( String aPath, String aName ) {
		super();
		_path = aPath;
		_name = aName;
	}

	/**
	 * Constructs a {@link FileHandle} object with the properties of the given
	 * {@link FileHandle}.
	 * 
	 * @param aFileHandle The {@link FileHandle} from which to take the required
	 *        properties.
	 */
	public FileHandleImpl( FileHandle aFileHandle ) {
		super();
		_path = aFileHandle.getPath();
		_name = aFileHandle.getName();
		_size = aFileHandle.getSize();
		_createdDate = aFileHandle.getCreatedDate();
		_modifiedDate = aFileHandle.getModifiedDate();
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public String getPath() {
		return _path;
	}

	@Override
	public String getName() {
		return _name;
	}

	@Override
	public String toKey() {
		return FileSystemUtility.toKey( _path, _name );
	}

	@Override
	public long getSize() {
		return _size;
	}

	@Override
	public Date getCreatedDate() {
		return _createdDate;
	}

	@Override
	public Date getModifiedDate() {
		return _modifiedDate;
	}

	// /////////////////////////////////////////////////////////////////////////
	// EQUALITY:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((_name == null) ? 0 : _name.hashCode());
		result = prime * result + ((_path == null) ? 0 : _path.hashCode());
		return result;
	}

	@Override
	public boolean equals( Object obj ) {
		if ( this == obj ) return true;
		if ( obj == null ) return false;
		// ---------------------------------------------------------------------
		// We treat a mutable file handle the same as an immutable file handle
		// in terms of equality. This makes live easier in collections though
		// complicates things when modifying attributes while the mutable file
		// handle is stored in a collection!
		// ---------------------------------------------------------------------
		if ( MutableFileHandleImpl.class != obj.getClass() && FileHandleImpl.class != obj.getClass() ) return false;
		FileHandle other = (FileHandle) obj;
		if ( _name == null ) {
			if ( other.getName() != null ) return false;
		}
		else if ( !_name.equals( other.getName() ) ) return false;
		if ( _path == null ) {
			if ( other.getPath() != null ) return false;
		}
		else if ( !_path.equals( other.getPath() ) ) return false;
		return true;
	}

	// /////////////////////////////////////////////////////////////////////////
	// MUTABLE:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public MutableFileHandle toMutableFileHandle() {
		return new MutableFileHandleImpl( this );
	}

	/**
	 * The implementation of a {@link MutableFileHandle}.
	 * -------------------------------------------------------------------------
	 * CAUTION: fiddling with the path and the name attributes causes the
	 * {@link #hashCode()} and {@link #equals(Object)} methods to change
	 * behavior which can cause problems especially in collections!
	 * -------------------------------------------------------------------------
	 * ATTENTION: In order to avoid the above mentioned problems with the
	 * {@link #equals(Object)} and {@link #hashCode()} methods, use
	 * {@link #toFileHandle()} before storing a {@link MutableFileHandle} in a
	 * collection.
	 */
	public class MutableFileHandleImpl extends FileHandleImpl implements MutableFileHandle {

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////


		public MutableFileHandleImpl( String aPath, String aName, long aSize, Date aCreatedDate, Date aModifiedDate ) {
			super( aPath, aName, aSize, aCreatedDate, aModifiedDate );
		}


		public MutableFileHandleImpl( FileHandle aFileHandle ) {
			super( aFileHandle );
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		@Override
		public void setModifiedDate( Date aModifiedDate ) {
			_modifiedDate = aModifiedDate;
		}

		@Override
		public void setSize( long aSize ) {
			_size = aSize;
		}

		@Override
		public void setName( String aName ) {
			_name = aName;
		}

		@Override
		public void setPath( String aPath ) {
			_path = aPath;
		}

		@Override
		public void setCreatedDate( Date aCreatedDate ) {
			_createdDate = aCreatedDate;
		}

		// /////////////////////////////////////////////////////////////////////////
		// EQUALITY:
		// /////////////////////////////////////////////////////////////////////////

		@Override
		public int hashCode() {
			return super.hashCode();
		}

		@Override
		public boolean equals( Object obj ) {
			return super.equals( obj );
		}

		// /////////////////////////////////////////////////////////////////////////
		// IMMUTABLE:
		// /////////////////////////////////////////////////////////////////////////

		@Override
		public FileHandle toFileHandle() {
			return new FileHandleImpl( this );
		}

	}
}
