// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.filesystem;

import org.refcodes.exception.AbstractException;
import org.refcodes.mixin.KeyAccessor;
import org.refcodes.mixin.NameAccessor;
import org.refcodes.mixin.PathAccessor;

/**
 * The Class FileSystemException.
 */
public abstract class FileSystemException extends AbstractException {

	private static final long serialVersionUID = 1L;

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param aMessage the message
	 * @param aErrorCode the error code
	 */
	public FileSystemException( String aMessage, String aErrorCode ) {
		super( aMessage, aErrorCode );
	}

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param aMessage the message
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public FileSystemException( String aMessage, Throwable aCause, String aErrorCode ) {
		super( aMessage, aCause, aErrorCode );
	}

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param message the message
	 * @param cause the cause
	 */
	public FileSystemException( String message, Throwable cause ) {
		super( message, cause );
	}

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param message the message
	 */
	public FileSystemException( String message ) {
		super( message );
	}

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public FileSystemException( Throwable aCause, String aErrorCode ) {
		super( aCause, aErrorCode );
	}

	/**
	 * Instantiates a new file system exception.
	 *
	 * @param cause the cause
	 */
	public FileSystemException( Throwable cause ) {
		super( cause );
	}

	/**
	 * The Class FileSystemFileException.
	 */
	static abstract class FileSystemFileException extends FileSystemException implements FileHandleAccessor {

		private static final long serialVersionUID = 1L;

		private FileHandle _file;

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public FileSystemFileException( FileHandle aFile, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_file = aFile;
		}

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemFileException( FileHandle aFile, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_file = aFile;
		}

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param message the message
		 * @param cause the cause
		 */
		public FileSystemFileException( FileHandle aFile, String message, Throwable cause ) {
			super( message, cause );
			_file = aFile;
		}

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param message the message
		 */
		public FileSystemFileException( FileHandle aFile, String message ) {
			super( message );
			_file = aFile;
		}

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemFileException( FileHandle aFile, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_file = aFile;
		}

		/**
		 * Instantiates a new file system file exception.
		 *
		 * @param aFile the file
		 * @param cause the cause
		 */
		public FileSystemFileException( FileHandle aFile, Throwable cause ) {
			super( cause );
			_file = aFile;
		}

		/**
		 * Gets the file.
		 *
		 * @return the file
		 */
		@Override
		public FileHandle getFile() {
			return _file;
		}

		/**
		 * The Class FileSystemNoFileAccessException.
		 */
		static abstract class FileSystemNoFileAccessException extends FileSystemFileException {

			private static final long serialVersionUID = 1L;

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, String aMessage, String aErrorCode ) {
				super( aFile, aMessage, aErrorCode );
			}

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, String aMessage, Throwable aCause, String aErrorCode ) {
				super( aFile, aMessage, aCause, aErrorCode );
			}

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, String message, Throwable cause ) {
				super( aFile, message, cause );
			}

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, String message ) {
				super( aFile, message );
			}

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, Throwable aCause, String aErrorCode ) {
				super( aFile, aCause, aErrorCode );
			}

			/**
			 * {@inheritDoc}
			 */
			public FileSystemNoFileAccessException( FileHandle aFile, Throwable cause ) {
				super( aFile, cause );
			}

		}
	}

	/**
	 * The Class FileSystemNoFileSystemAccessException.
	 */
	static abstract class FileSystemNoFileSystemAccessException extends FileSystemException {

		private static final long serialVersionUID = 1L;

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public FileSystemNoFileSystemAccessException( String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
		}

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemNoFileSystemAccessException( String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
		}

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param message the message
		 * @param cause the cause
		 */
		public FileSystemNoFileSystemAccessException( String message, Throwable cause ) {
			super( message, cause );
		}

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param message the message
		 */
		public FileSystemNoFileSystemAccessException( String message ) {
			super( message );
		}

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemNoFileSystemAccessException( Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
		}

		/**
		 * Instantiates a new file system no file system access exception.
		 *
		 * @param cause the cause
		 */
		public FileSystemNoFileSystemAccessException( Throwable cause ) {
			super( cause );
		}
	}

	/**
	 * The Class FileSystemKeyException.
	 */
	static abstract class FileSystemKeyException extends FileSystemException implements KeyAccessor<String> {

		private static final long serialVersionUID = 1L;

		private String _key;

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public FileSystemKeyException( String aKey, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_key = aKey;
		}

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemKeyException( String aKey, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_key = aKey;
		}

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param message the message
		 * @param cause the cause
		 */
		public FileSystemKeyException( String aKey, String message, Throwable cause ) {
			super( message, cause );
			_key = aKey;
		}

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param message the message
		 */
		public FileSystemKeyException( String aKey, String message ) {
			super( message );
			_key = aKey;
		}

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemKeyException( String aKey, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_key = aKey;
		}

		/**
		 * Instantiates a new file system key exception.
		 *
		 * @param aKey the key
		 * @param cause the cause
		 */
		public FileSystemKeyException( String aKey, Throwable cause ) {
			super( cause );
			_key = aKey;
		}

		/**
		 * Gets the key.
		 *
		 * @return the key
		 */
		@Override
		public String getKey() {
			return _key;
		}
	}

	/**
	 * The Class FileSystemNameException.
	 */
	static abstract class FileSystemNameException extends FileSystemException implements NameAccessor {

		private static final long serialVersionUID = 1L;

		private String _name;

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public FileSystemNameException( String aName, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_name = aName;
		}

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemNameException( String aName, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_name = aName;
		}

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param message the message
		 * @param cause the cause
		 */
		public FileSystemNameException( String aName, String message, Throwable cause ) {
			super( message, cause );
			_name = aName;
		}

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param message the message
		 */
		public FileSystemNameException( String aName, String message ) {
			super( message );
			_name = aName;
		}

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemNameException( String aName, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_name = aName;
		}

		/**
		 * Instantiates a new file system name exception.
		 *
		 * @param aName the name
		 * @param cause the cause
		 */
		public FileSystemNameException( String aName, Throwable cause ) {
			super( cause );
			_name = aName;
		}

		/**
		 * Gets the name.
		 *
		 * @return the name
		 */
		@Override
		public String getName() {
			return _name;
		}
	}

	/**
	 * The Class FileSystemPathException.
	 */
	static abstract class FileSystemPathException extends FileSystemException implements PathAccessor {

		private static final long serialVersionUID = 1L;

		private String _path;

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public FileSystemPathException( String aPath, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_path = aPath;
		}

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemPathException( String aPath, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_path = aPath;
		}

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param message the message
		 * @param cause the cause
		 */
		public FileSystemPathException( String aPath, String message, Throwable cause ) {
			super( message, cause );
			_path = aPath;
		}

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param message the message
		 */
		public FileSystemPathException( String aPath, String message ) {
			super( message );
			_path = aPath;
		}

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public FileSystemPathException( String aPath, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_path = aPath;
		}

		/**
		 * Instantiates a new file system path exception.
		 *
		 * @param aPath the path
		 * @param cause the cause
		 */
		public FileSystemPathException( String aPath, Throwable cause ) {
			super( cause );
			_path = aPath;
		}

		/**
		 * Gets the path.
		 *
		 * @return the path
		 */
		@Override
		public String getPath() {
			return _path;
		}
	}
}
