// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.forwardsecrecy;

import org.refcodes.data.Text;

/**
 * Default implementation of a cipher version generator.
 *
 * @param <CV> The type of the {@link CipherVersion} to be generated.
 */
public abstract class AbstractCipherVersionGenerator<CV extends CipherVersion> implements CipherVersionGenerator<CV> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private CipherVersionFactory<CV> _cipherVersionFactory;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructor for the {@link AbstractCipherVersionGenerator}. The provided
	 * {@link CipherVersionFactory} is used to create the actual
	 * {@link CipherVersion} instances generated by this
	 * {@link CipherVersionGenerator}.
	 * 
	 * @param aCipherVersionFactory The {@link CipherVersionFactory} used to
	 *        create the {@link CipherVersion} instances of the required type.
	 */
	public AbstractCipherVersionGenerator( CipherVersionFactory<CV> aCipherVersionFactory ) {
		_cipherVersionFactory = aCipherVersionFactory;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasNext() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove() {
		throw new UnsupportedOperationException( Text.UNSUPPORTED_OPERATION.getText() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CV next() {
		return _cipherVersionFactory.createInstance( createCipherUid(), createCipher() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOK METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Hook factory method to be implemented by subclasses. A special algorithm
	 * can be used to generate unique cipher UIDs. In case the cipher UID is not
	 * unique, another one is requested until a cipher UID is found which is
	 * unique regarding the systems of forward secrecy cryptography
	 * architecture.
	 * 
	 * @return A more or less unique cipher UID.
	 */
	protected String createCipherUid() {
		return ForwardSecrecyUtility.createCipherUid();
	}

	/**
	 * Hook factory method to be implemented by subclasses. A special algorithm
	 * is to be implemented creating a good random cipher.
	 * 
	 * @return A good random cipher.
	 */
	protected String createCipher() {
		return ForwardSecrecyUtility.createCipher();
	}
}
