// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.forwardsecrecy;

import org.refcodes.controlflow.RetryCounter;
import org.refcodes.controlflow.RetryCounterImpl;
import org.refcodes.data.LoopSleepTime;
import org.refcodes.data.RetryCount;
import org.refcodes.data.Text;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;

/**
 * Abstract base implementation for non abstract {@link EncryptionService}
 * implementations.
 */
public abstract class AbstractEncryptionService implements EncryptionService {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private CipherVersionGenerator<CipherVersion> _cipherVersionGenerator;

	private String _namespace;

	private EncryptionServer _encryptionServer;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the service with the required services and configuration.
	 * 
	 * @param aNamespace The name space to which this service belongs
	 * 
	 * @param aEncryptionServer The server to which the service is being
	 *        "connected"
	 * 
	 * @param aCipherVersionGenerator The generator to be used for generating
	 *        {@link CipherVersion} instances.
	 */
	public AbstractEncryptionService( String aNamespace, EncryptionServer aEncryptionServer, CipherVersionGenerator<CipherVersion> aCipherVersionGenerator ) {
		_cipherVersionGenerator = aCipherVersionGenerator;
		_encryptionServer = aEncryptionServer;
		_namespace = aNamespace;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CipherVersion next() {
		CipherVersion eCipherVersion = null;
		RetryCounter theRetryCounter = new RetryCounterImpl( RetryCount.NORM.getNumber(), LoopSleepTime.MIN.getMilliseconds() );
		while ( theRetryCounter.nextRetry() ) {
			try {
				eCipherVersion = _cipherVersionGenerator.next();

				// -------------------------------------------------------------
				// Hotspot for persisting the cipher in a DB or a storage using
				// some kind of asymmetric encryption mechanism
				// -------------------------------------------------------------
				_encryptionServer.addCipherVersion( _namespace, toEncryptedCipherVersion( eCipherVersion ) );
				return eCipherVersion;
			}
			catch ( CipherUidAlreadyInUseException e ) {
				if ( !theRetryCounter.hasNextRetry() ) {
					// @formatter:off
					LOGGER.warn( "Failed to add a cipher UID \"" + eCipherVersion.getUniversalId() + "\", retry count is <" + theRetryCounter.getRetryCount() + "> of <" + theRetryCounter.getRetryNumber() + "> (waiting for <" + ( theRetryCounter.getNextRetryDelayInMs() / 1000 ) + "> seconds before next retry): \"" + ExceptionUtility.toMessage( e ) + "\"" );
					// @formatter:on

				}
			}
		}
		throw new UnexpectedForwardSecrecyRuntimeException( "Unable to generate a Chipher-Version within <" + theRetryCounter.getRetryNumber() + "> tries, aborting as else we might loop infinte time!" );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasNext() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove() {
		throw new UnsupportedOperationException( Text.UNSUPPORTED_OPERATION.getText() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOK METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Hook factory method to be implemented by subclasses. The provided cipher
	 * is to be encrypted (e.g. with the public key of an asymmetric encryption
	 * approach) so that an Encryption-Server only receives encrypted data.
	 * 
	 * @param <CV> The type of the {@link CipherVersion} to be used.
	 * 
	 * @param aDecyrptedCipherVersion The {@link CipherVersion} to be encrypted.
	 * 
	 * @return The encrypted {@link CipherVersion}.
	 */
	protected abstract <CV extends CipherVersion> CV toEncryptedCipherVersion( CV aDecyrptedCipherVersion );
}
