// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.forwardsecrecy;

import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

import org.jasypt.util.text.BasicTextEncryptor;
import org.refcodes.data.Delimiter;
import org.refcodes.data.License;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.textual.HorizAlignTextBuilderImpl;
import org.refcodes.textual.HorizAlignTextMode;
import org.refcodes.textual.RandomTextGenerartor;
import org.refcodes.textual.RandomTextGenerartorImpl;
import org.refcodes.textual.RandomTextMode;

/**
 * The Class ForwardSecrecyUtility.
 */
public final class ForwardSecrecyUtility {

	private static final char CIPHER_UID_TIMESTAMP_SEPARATOR = '-';

	public static final int CIPHER_UID_TIMESTAMP_LENGTH = 14;

	public static final int CIPHER_UID_LENGTH = 24;

	public static final int CIPHER_LENGTH = 48;

	public static final int MESSAGE_LENGTH = 256;

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	private static BasicTextEncryptor TEXT_ENCRYPTOR;

	private static RandomTextGenerartor RND_CIPHER_GENERATOR = new RandomTextGenerartorImpl().withColumnWidth( CIPHER_LENGTH ).withRandomTextMode( RandomTextMode.ALPHANUMERIC );

	private static RandomTextGenerartor RND_MESSAGE_GENERATOR = new RandomTextGenerartorImpl().withColumnWidth( MESSAGE_LENGTH ).withRandomTextMode( RandomTextMode.ALPHANUMERIC );

	static {
		String thePassPhrase = null;
		Enumeration<NetworkInterface> e;
		try {
			e = NetworkInterface.getNetworkInterfaces();
			NetworkInterface eNetworkInterface;
			byte[] eHardwareAddress;
			while ( e.hasMoreElements() ) {
				eNetworkInterface = e.nextElement();
				try {
					eHardwareAddress = eNetworkInterface.getHardwareAddress();
					if ( eHardwareAddress != null ) {
						thePassPhrase = "";
						for ( int i = 0; i < eHardwareAddress.length; i++ ) {
							thePassPhrase += (eHardwareAddress[i] < 0) ? "" + ((eHardwareAddress[i]) & 0xFF) : "" + eHardwareAddress[i];
							if ( i < eHardwareAddress.length - 1 ) {
								thePassPhrase += ".";
							}
						}
						break;
					}
				}
				catch ( SocketException se ) {
					LOGGER.warn( "Unable to aquire network interfaces's (\"" + eNetworkInterface.getDisplayName() + "\") hardware address, tyring next network interface.", se );
				}
			}
		}
		catch ( SocketException se ) {
			LOGGER.warn( "Unable to aquire machine's network interfaces (using alternate cipher).", se );
		}

		if ( thePassPhrase == null ) {
			thePassPhrase = License.REFCODES_LICENSE.getText();
		}

		TEXT_ENCRYPTOR = new BasicTextEncryptor();
		TEXT_ENCRYPTOR.setPassword( thePassPhrase );
	}

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private ForwardSecrecyUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns true in case the given text provided the characteristics of an
	 * encrypted text as of the cipher and cipher UID pattern.
	 * -------------------------------------------------------------------------
	 * CAUTION: A plain text may have the same characteristics, an encrypted
	 * text must have these characteristics!
	 * -------------------------------------------------------------------------
	 * 
	 * @param aText The text to test whether it is encrypted.
	 * 
	 * @return True in case the given text has the characteristics of an
	 *         encrypted text
	 */
	public static boolean hasEncryptionPattern( String aText ) {
		if ( aText.length() < CIPHER_UID_LENGTH ) {
			return false;
		}
		if ( aText.charAt( CIPHER_UID_LENGTH ) != Delimiter.CIPHER_UID.getChar() ) {
			return false;
		}
		if ( aText.charAt( CIPHER_UID_TIMESTAMP_LENGTH - 1 ) != CIPHER_UID_TIMESTAMP_SEPARATOR ) {
			return false;
		}
		return true;
	}

	/**
	 * Expects a text with a prefixed cipher UID. Extracts the cipher UID with
	 * which the given text was encrypted.
	 * 
	 * @param aCipherUidWithEncryptedText The encrypted text with the prefixed
	 *        cipher UID
	 * 
	 * @return The cipher UID or null if none cipher UID was found
	 */
	public static String toCipherUidPrefix( String aCipherUidWithEncryptedText ) {
		int theIndex = aCipherUidWithEncryptedText.indexOf( Delimiter.CIPHER_UID.getChar() );
		if ( theIndex == -1 ) {
			return null;
		}
		return aCipherUidWithEncryptedText.substring( 0, theIndex );
	}

	/**
	 * Expects a text with a prefixed cipher UID. Extracts the encrypted Text
	 * without the prefixed cipher UID.
	 * 
	 * @param aCipherUidWithEncryptedText The encrypted text with the prefixed
	 *        cipher UID
	 * 
	 * @return The encrypted text portion or null if none cipher UID was found
	 *         (then we have an invalid format of the provided text)
	 */
	public static String toEncryptedTextBody( String aCipherUidWithEncryptedText ) {
		int theIndex = aCipherUidWithEncryptedText.indexOf( Delimiter.CIPHER_UID.getChar() );
		if ( theIndex == -1 ) {
			return null;
		}
		return aCipherUidWithEncryptedText.substring( theIndex + 1 );
	}

	/**
	 * Default way on how to create a cipher UID. In case the default way
	 * generated bad cipher UIDs, the default way's implementation is changed
	 * making it good again and all system using the default way.
	 * 
	 * The timestamp is prepended so that regarding on the timestamp, encrypted
	 * texts can be easily selected, e.g. texts being encrypted with a cipher
	 * older than a given timestamp.
	 * 
	 * @return A cipher UID created the default way.
	 */
	public static String createCipherUid() {
		String theCipherUid;
		theCipherUid = System.currentTimeMillis() + "" + CIPHER_UID_TIMESTAMP_SEPARATOR;
		theCipherUid = new HorizAlignTextBuilderImpl().withHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withText( theCipherUid ).withColumnWidth( CIPHER_UID_TIMESTAMP_LENGTH ).withFillChar( '0' ).toString();
		theCipherUid = theCipherUid + new RandomTextGenerartorImpl().withColumnWidth( CIPHER_UID_LENGTH - theCipherUid.length() ).withRandomTextMode( RandomTextMode.ALPHANUMERIC ).next();
		return theCipherUid;
	}

	/**
	 * Default way on how to create a cipher. In case the default way generated
	 * bad ciphers, the default way's implementation is changed making it secure
	 * again and all system using the default way.
	 * 
	 * @return A cipher created the default way.
	 */
	public static String createCipher() {
		return RND_CIPHER_GENERATOR.next();
	}

	/**
	 * Default way on how to create a message which is to be signed in order to
	 * identify the owner of a public key.
	 * 
	 * @return A message created the default way.
	 */
	public static String createMessage() {
		return RND_MESSAGE_GENERATOR.next();
	}

}
