// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.forwardsecrecy;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.PrivateKey;

import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;

import edu.vt.middleware.crypt.CryptException;
import edu.vt.middleware.crypt.asymmetric.AsymmetricAlgorithm;
import edu.vt.middleware.crypt.asymmetric.RSA;
import edu.vt.middleware.crypt.digest.SHA512;
import edu.vt.middleware.crypt.signature.RSASignature;
import edu.vt.middleware.crypt.signature.SignatureAlgorithm;
import edu.vt.middleware.crypt.util.Base64Converter;
import edu.vt.middleware.crypt.util.CryptReader;

/**
 * The {@link DecryptionService} retrieves {@link CipherVersion} instances from
 * the {@link DecryptionServer} and is decrypting the ciphers contained in the
 * {@link CipherVersion} instances with a private key.
 */
public class PublicKeyDecryptionServiceImpl extends AbstractDecryptionService {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private CipherVersionFactory<CipherVersion> _cipherVersionFactory;

	private AsymmetricAlgorithm _decryptAlgorithm = new RSA();

	private Base64Converter _base64Converter = new Base64Converter();

	private SignatureAlgorithm _signatureAlgorithm = new RSASignature( new SHA512() );

	private String _privateKeyPath; /* For logging purposes only */

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the service with the required services and configuration.
	 * 
	 * {@inheritDoc}
	 * 
	 * @param aPrivateKeyPath The path on the file system to the private key to
	 *        be used for decrypting any ciphers passed by the decryption server
	 *        in CipherVersions retrieved from a storage.
	 * 
	 * @param aDecryptionServer The decryption server for accessing the cipher
	 *        versions
	 * 
	 * @param aCipherVersionFactory The factory to be used for creating
	 *        {@link CipherVersion} instances.
	 * 
	 * @throws IOException in case of I/O problems
	 * @throws CryptException in case the cryptography algorithm had problems.
	 */
	public PublicKeyDecryptionServiceImpl( String aNamespace, String aPrivateKeyPath, DecryptionServer aDecryptionServer, CipherVersionFactory<CipherVersion> aCipherVersionFactory ) throws CryptException, IOException {
		this( aNamespace, aPrivateKeyPath, null, aDecryptionServer, aCipherVersionFactory );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 * 
	 * {@inheritDoc}
	 * 
	 * @param aPrivateKeyPath The path on the file system to the private key to
	 *        be used for decrypting any ciphers passed by the decryption server
	 *        in CipherVersions retrieved from a storage.
	 * 
	 * @param aDecryptionServer The decryption server for accessing the cipher
	 *        versions
	 * 
	 * @throws IOException in case of I/O problems
	 * @throws CryptException in case the cryptography algorithm had problems.
	 */
	public PublicKeyDecryptionServiceImpl( String aNamespace, String aPrivateKeyPath, DecryptionServer aDecryptionServer ) throws CryptException, IOException {
		this( aNamespace, aPrivateKeyPath, null, aDecryptionServer, new CipherVersionFactoryImpl() );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 * 
	 * {@inheritDoc}
	 * 
	 * @param aPrivateKeyPath The path on the file system to the private key to
	 *        be used for decrypting any ciphers passed by the decryption server
	 *        in CipherVersions retrieved from a storage.
	 * 
	 * @param aPrivateKeyPassPhrase The pass phrase for decrypting the private
	 *        key.
	 * 
	 * @param aDecryptionServer The decryption server for accessing the cipher
	 *        versions
	 * 
	 * @throws IOException in case of I/O problems
	 * @throws CryptException in case the cryptography algorithm had problems.
	 */
	public PublicKeyDecryptionServiceImpl( String aNamespace, String aPrivateKeyPath, String aPrivateKeyPassPhrase, DecryptionServer aDecryptionServer ) throws CryptException, IOException {
		this( aNamespace, aPrivateKeyPath, aPrivateKeyPassPhrase, aDecryptionServer, new CipherVersionFactoryImpl() );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 * 
	 * {@inheritDoc}
	 * 
	 * @param aPrivateKeyPath The path on the file system to the private key to
	 *        be used for decrypting any ciphers passed by the decryption server
	 *        in CipherVersions retrieved from a storage.
	 * 
	 * @param aPrivateKeyPassPhrase The pass phrase for decrypting the private
	 *        key.
	 * 
	 * @param aDecryptionServer The decryption server for accessing the cipher
	 *        versions
	 * 
	 * @param aCipherVersionFactory The factory to be used for creating
	 *        {@link CipherVersion} instances.
	 * 
	 * @throws IOException in case of I/O problems
	 * @throws CryptException in case the cryptography algorithm had problems.
	 */
	public PublicKeyDecryptionServiceImpl( String aNamespace, String aPrivateKeyPath, String aPrivateKeyPassPhrase, DecryptionServer aDecryptionServer, CipherVersionFactory<CipherVersion> aCipherVersionFactory ) throws CryptException, IOException {
		super( aNamespace, aDecryptionServer );
		_cipherVersionFactory = aCipherVersionFactory;
		_privateKeyPath = aPrivateKeyPath;
		File thePrivateKeyFile = new File( aPrivateKeyPath );
		LOGGER.debug( "Loading private key from file \"" + thePrivateKeyFile.getAbsolutePath() + "\"..." );
		try {
			PrivateKey thePrivateKey;
			if ( aPrivateKeyPassPhrase != null ) {
				thePrivateKey = CryptReader.readPrivateKey( thePrivateKeyFile, aPrivateKeyPassPhrase.toCharArray() );
			}
			else {
				thePrivateKey = CryptReader.readPrivateKey( thePrivateKeyFile );
			}
			_decryptAlgorithm.setKey( thePrivateKey );
			_decryptAlgorithm.initDecrypt();
			_signatureAlgorithm.setSignKey( thePrivateKey );
			_signatureAlgorithm.initSign();
		}
		catch ( FileNotFoundException e ) {
			LOGGER.error( "Unable to load private key from file \"" + thePrivateKeyFile.getAbsolutePath() + "\": " + ExceptionUtility.toMessage( e ) + "\"" );
			throw e;
		}
		catch ( CryptException e ) {
			// @formatter:off
			LOGGER.error( "Unable to intantiate private key from file   \"" + thePrivateKeyFile.getAbsolutePath() + " possibly due to a \"Java Cryptography Extension (JCE)\" with limited Strength (in that case please update to \"Java Cryptography Extension (JCE) Unlimited Strength\") \": " + ExceptionUtility.toMessage( e ) + "\"" );
			// @formatter:on
			throw e;
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOK METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String toSignature( String aMessage ) {
		LOGGER.debug( "Using private key \"" + _privateKeyPath + "\" for signing a message used to identify a public key ..." );
		try {
			return _signatureAlgorithm.sign( aMessage.getBytes(), _base64Converter );
		}
		catch ( CryptException e ) {
			throw new HiddenException( e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String createMessage() {
		return ForwardSecrecyUtility.createMessage();
	}

	/**
	 * To decrypted cipher version.
	 *
	 * @param <CV> the generic type
	 * @param aEncyrptedCipherVersion the encyrpted cipher version
	 * @return the cv
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected <CV extends CipherVersion> CV toDecryptedCipherVersion( CV aEncyrptedCipherVersion ) {
		LOGGER.debug( "Using private key \"" + _privateKeyPath + "\" for decrypting a cipher ..." );
		try {
			return (CV) _cipherVersionFactory.createInstance( aEncyrptedCipherVersion.getUniversalId(), new String( _decryptAlgorithm.decrypt( aEncyrptedCipherVersion.getCipher(), _base64Converter ) ) );
		}
		catch ( CryptException e ) {
			throw new HiddenException( e );
		}
	}
}
