// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.forwardsecrecy;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.PublicKey;

import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;

import edu.vt.middleware.crypt.CryptException;
import edu.vt.middleware.crypt.asymmetric.AsymmetricAlgorithm;
import edu.vt.middleware.crypt.asymmetric.RSA;
import edu.vt.middleware.crypt.util.Base64Converter;
import edu.vt.middleware.crypt.util.CryptReader;

/**
 * This Encryption-Service uses a public key for an asymmetric encryption
 * algorithm in order to encrypt any ciphers being generated before them ciphers
 * are passed to the Encryption-Server which persists them cipher versions with
 * the cipher UID and the encrypted cipher. A decryption server may retrieve the
 * cipher versions with the cipher UID and the encrypted cipher and pass it to a
 * Decryption-Service which might decrypt the ciphers with an according private
 * key.
 */
public class PublicKeyEncryptionServiceImpl extends AbstractEncryptionService {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private CipherVersionFactory<CipherVersion> _cipherVersionFactory;

	private AsymmetricAlgorithm _encryptAlgorithm = new RSA();

	private Base64Converter _base64Converter = new Base64Converter();

	private String _publicKeyPath; /* For logging only */

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the service with the required services and configuration.
	 *
	 * @param aNamespace The name space to which this service belongs
	 * @param aPublicKeyPath The path on the file system to the public key to be
	 *        used for encrypting any ciphers passed to the Encryption-Server
	 *        for persisting.
	 * @param aEncryptionServer The server to which the service is being
	 *        "connected"
	 * @throws CryptException in case a crypt exception occurred when
	 *         instantiating the public key
	 * @throws IOException in case reading the file public key caused IO
	 *         problems
	 */
	public PublicKeyEncryptionServiceImpl( String aNamespace, String aPublicKeyPath, EncryptionServer aEncryptionServer ) throws CryptException, IOException {
		this( aNamespace, aPublicKeyPath, aEncryptionServer, new CipherVersionGeneratorImpl(), new CipherVersionFactoryImpl() );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 *
	 * @param aNamespace The name space to which this service belongs
	 * @param aPublicKeyPath The path on the file system to the public key to be
	 *        used for encrypting any ciphers passed to the Encryption-Server
	 *        for persisting.
	 * @param aEncryptionServer The server to which the service is being
	 *        "connected"
	 * @param aCipherVersionGenerator The generator to be used for generating
	 *        {@link CipherVersion} instances.
	 * @throws CryptException in case a crypt exception occurred when
	 *         instantiating the public key
	 * @throws IOException in case reading the file public key caused IO
	 *         problems
	 */
	public PublicKeyEncryptionServiceImpl( String aNamespace, String aPublicKeyPath, EncryptionServer aEncryptionServer, CipherVersionGenerator<CipherVersion> aCipherVersionGenerator ) throws CryptException, IOException {
		this( aNamespace, aPublicKeyPath, aEncryptionServer, aCipherVersionGenerator, new CipherVersionFactoryImpl() );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 *
	 * @param aNamespace The name space to which this service belongs
	 * @param aPublicKeyPath The path on the file system to the public key to be
	 *        used for encrypting any ciphers passed to the Encryption-Server
	 *        for persisting.
	 * @param aEncryptionServer The server to which the service is being
	 *        "connected"
	 * @param aCipherVersionFactory The factory to be used for creating
	 *        {@link CipherVersion} instances.
	 * @throws CryptException in case a crypt exception occurred when
	 *         instantiating the public key
	 * @throws IOException in case reading the file public key caused IO
	 *         problems
	 */
	public PublicKeyEncryptionServiceImpl( String aNamespace, String aPublicKeyPath, EncryptionServer aEncryptionServer, CipherVersionFactory<CipherVersion> aCipherVersionFactory ) throws CryptException, IOException {
		this( aNamespace, aPublicKeyPath, aEncryptionServer, new CipherVersionGeneratorImpl(), aCipherVersionFactory );
	}

	/**
	 * Constructs the service with the required services and configuration.
	 *
	 * @param aNamespace The name space to which this service belongs
	 * @param aPublicKeyPath The path on the file system to the public key to be
	 *        used for encrypting any ciphers passed to the Encryption-Server
	 *        for persisting.
	 * @param aEncryptionServer The server to which the service is being
	 *        "connected"
	 * @param aCipherVersionGenerator The generator to be used for generating
	 *        {@link CipherVersion} instances.
	 * @param aCipherVersionFactory The factory to be used for creating
	 *        {@link CipherVersion} instances.
	 * @throws CryptException in case a crypt exception occurred when
	 *         instantiating the public key
	 * @throws IOException in case reading the file public key caused IO
	 *         problems
	 */
	public PublicKeyEncryptionServiceImpl( String aNamespace, String aPublicKeyPath, EncryptionServer aEncryptionServer, CipherVersionGenerator<CipherVersion> aCipherVersionGenerator, CipherVersionFactory<CipherVersion> aCipherVersionFactory ) throws CryptException, IOException {
		super( aNamespace, aEncryptionServer, aCipherVersionGenerator );
		_cipherVersionFactory = aCipherVersionFactory;
		_publicKeyPath = aPublicKeyPath;
		File thePublicKeyFile = new File( aPublicKeyPath );
		LOGGER.debug( "Loading public key from file \"" + thePublicKeyFile.getAbsolutePath() + "\"..." );
		try {
			PublicKey thePublicKey = CryptReader.readPublicKey( thePublicKeyFile );
			_encryptAlgorithm.setKey( thePublicKey );
			_encryptAlgorithm.initEncrypt();
		}
		catch ( FileNotFoundException e ) {
			LOGGER.warn( "Unable to load public key from file \"" + thePublicKeyFile.getAbsolutePath() + "\": " + ExceptionUtility.toMessage( e ) );
			throw e;
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOK METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To encrypted cipher version.
	 *
	 * @param <CV> the generic type
	 * @param aDecryptedCipherVersion the decrypted cipher version
	 * @return the cv
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected <CV extends CipherVersion> CV toEncryptedCipherVersion( CV aDecryptedCipherVersion ) {
		LOGGER.debug( "Using public key \"" + _publicKeyPath + "\" for encrypting ..." );
		try {
			return (CV) _cipherVersionFactory.createInstance( aDecryptedCipherVersion.getUniversalId(), _encryptAlgorithm.encrypt( aDecryptedCipherVersion.getCipher().getBytes(), _base64Converter ) );
		}
		catch ( CryptException e ) {
			throw new HiddenException( e );
		}
	}
}
