// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.graphical;

/**
 * Provides an accessor for a {@link GridDimension} property.
 */
public interface GridDimension extends GridHeightAccessor, GridWidthAccessor {

	/**
	 * Tests whether the given position is inside of the bounds of the grid. The
	 * x and y coordinates must be greater than or equals to 0 and smaller than
	 * the grid width (x) and grid height (y).
	 * 
	 * @param aPosition The position for which to determine if it is in bounds.
	 * 
	 * @return True in case the given position is in bounds.
	 */
	default boolean isInBounds( Position aPosition ) {
		return isInBounds( aPosition.getPositionX(), aPosition.getPositionY() );
	}

	/**
	 * Tests whether the given position is inside of the bounds of the grid. The
	 * x- and y-coordinates must be greater or equals to 0 and smaller than the
	 * grid width (x) and grid height (y).
	 * 
	 * @param aPositionX The x-position for which to determine if it is in
	 *        bounds regarding the grid width.
	 * 
	 * @param aPositionY The y-position for which to determine if it is in
	 *        bounds regarding the grid height.
	 * 
	 * @return True in case the given position is in bounds.
	 */
	default boolean isInBounds( int aPositionX, int aPositionY ) {
		return aPositionX < getGridWidth() && aPositionY < getGridHeight() && aPositionX >= 0 && aPositionY >= 0;
	}

	/**
	 * Tests whether the given x-position is inside of the bounds of the grid's
	 * width. The x-coordinate must be greater than or equals to 0 and smaller
	 * than the grid width.
	 * 
	 * @param aPositionX The x-position for which to determine if it is in
	 *        bounds regarding the grid width.
	 * 
	 * @return True in case the given x-position is in bounds.
	 */
	default boolean isInWidth( int aPositionX ) {
		return aPositionX < getGridWidth() && aPositionX >= 0;
	}

	/**
	 * Tests whether the given y-position is inside of the bounds of the grid's
	 * height. The y-coordinate must be greater than or equals to 0 and smaller
	 * than the grid height.
	 * 
	 * @param aPositionY The y-position for which to determine if it is in
	 *        bounds regarding the grid height.
	 * 
	 * @return True in case the given y-position is in bounds.
	 */
	default boolean isInHeight( int aPositionY ) {
		return aPositionY < getGridHeight() && aPositionY >= 0;
	}

	/**
	 * The Interface GridDimensionMutator.
	 */
	public interface GridDimensionMutator extends GridHeightMutator, GridWidthMutator {

		/**
		 * Sets the grid dimension.
		 *
		 * @param aWidth the width
		 * @param aHeight the height
		 */
		void setGridDimension( int aWidth, int aHeight );

		/**
		 * Sets the grid dimension.
		 *
		 * @param aDimension the new grid dimension
		 */
		void setGridDimension( GridDimension aDimension );

		/**
		 * Sets the grid dimension.
		 *
		 * @param aDimension the new grid dimension
		 */
		void setGridDimension( Dimension aDimension );
	}

	/**
	 * The Interface GridDimensionBuilder.
	 *
	 * @param <B> the generic type
	 */
	public interface GridDimensionBuilder<B extends GridDimensionBuilder<B>> extends GridWidthBuilder<B>, GridHeightBuilder<B> {

		/**
		 * With grid dimension.
		 *
		 * @param aGridWidth the grid width
		 * @param aGridHeight the grid height
		 * @return the b
		 */
		B withGridDimension( int aGridWidth, int aGridHeight );

		/**
		 * With grid dimension.
		 *
		 * @param aDimension the dimension
		 * @return the b
		 */
		B withGridDimension( GridDimension aDimension );

		/**
		 * With grid dimension.
		 *
		 * @param aDimension the dimension
		 * @return the b
		 */
		B withGridDimension( Dimension aDimension );
	}

	/**
	 * The Interface GridDimensionProperty.
	 */
	public interface GridDimensionProperty extends GridDimension, GridDimensionMutator, GridWidthProperty, GridHeightProperty {}

	/**
	 * The Interface GridDimensionPropertyBuilder.
	 */
	public interface GridDimensionPropertyBuilder extends GridDimensionProperty, GridDimensionBuilder<GridDimensionPropertyBuilder> {}

	/**
	 * Equals.
	 *
	 * @param aGridDimensionA the grid dimension A
	 * @param aGridDimensionB the grid dimension B
	 * @return true, if successful
	 */
	static boolean equals( GridDimension aGridDimensionA, GridDimension aGridDimensionB ) {
		return aGridDimensionA.getGridWidth() == aGridDimensionB.getGridWidth() && aGridDimensionA.getGridHeight() == aGridDimensionB.getGridHeight();
	}

}
