// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.graphical;

/**
 * Provides an accessor for a Y position property.
 */
public interface Position extends PosXAccessor, PosYAccessor {

	/**
	 * Adds the given {@link Position} to this {@link Position} (does not modify
	 * this {@link Position}) and returns the resulting {@link Position}.
	 * 
	 * @param aPosition The position to be added.
	 * @return The resulting position.
	 */
	default Position toAddTo( Position aPosition ) {
		return new PositionImpl( aPosition.getPositionX() + getPositionX(), aPosition.getPositionY() + getPositionY() );
	}

	/**
	 * Subtracts the given {@link Position} from this {@link Position} (does not
	 * modify this {@link Position}) and returns the resulting {@link Position}.
	 * 
	 * @param aPosition The position to be subtracted.
	 * @return The resulting position.
	 */
	default Position toSubFrom( Position aPosition ) {
		return new PositionImpl( getPositionX() - aPosition.getPositionX(), getPositionY() - aPosition.getPositionY() );
	}

	/**
	 * The Interface PositionMutator.
	 */
	public interface PositionMutator extends PosXMutator, PosYMutator {

		/**
		 * Sets the position.
		 *
		 * @param aPosX the pos X
		 * @param aPosY the pos Y
		 */
		void setPosition( int aPosX, int aPosY );

		/**
		 * Sets the position.
		 *
		 * @param aPosition the new position
		 */
		void setPosition( Position aPosition );
	}

	/**
	 * The Interface PositionBuilder.
	 *
	 * @param <B> the generic type
	 */
	public interface PositionBuilder<B extends PositionBuilder<B>> extends PosXBuilder<B>, PosYBuilder<B>, PositionMutator {

		/**
		 * With position.
		 *
		 * @param aPosX the pos X
		 * @param aPosY the pos Y
		 * @return the b
		 */
		B withPosition( int aPosX, int aPosY );

		/**
		 * With position.
		 *
		 * @param aPosition the position
		 * @return the b
		 */
		B withPosition( Position aPosition );
	}

	/**
	 * The Interface PositionProperty.
	 */
	public interface PositionProperty extends Position, PositionMutator, PosXProperty, PosYProperty {

		/**
		 * Adds the given {@link Position} to this {@link Position} (does not
		 * modify this {@link Position}).
		 * 
		 * @param aPosition The position to be added.
		 */
		default void addTo( Position aPosition ) {
			setPosition( aPosition.getPositionX() + getPositionX(), aPosition.getPositionY() + getPositionY() );
		}

		/**
		 * Subtracts the given {@link Position} from this {@link Position} (does
		 * modify this {@link Position}).
		 * 
		 * @param aPosition The position to be subtracted.
		 */
		default void subFrom( Position aPosition ) {
			setPosition( getPositionX() - aPosition.getPositionX(), getPositionY() - aPosition.getPositionY() );
		}

	}

	/**
	 * The Interface PositionPropertyBuilder.
	 */
	public interface PositionPropertyBuilder extends PositionProperty, PositionBuilder<PositionPropertyBuilder> {}

	/**
	 * Equals.
	 *
	 * @param aPositionA the position A
	 * @param aPositionB the position B
	 * @return true, if successful
	 */
	static boolean equals( Position aPositionA, Position aPositionB ) {
		return aPositionA.getPositionX() == aPositionB.getPositionX() && aPositionA.getPositionY() == aPositionB.getPositionY();
	}
}
