// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.graphical;

/**
 * Provides an accessor for a grid width property.
 */
public interface GridWidthAccessor {

	/**
	 * Retrieves the grid width from the grid width property.
	 * 
	 * @return The grid width stored by the grid width property.
	 */
	int getGridWidth();

	/**
	 * Provides a mutator for a grid width property.
	 */
	public interface GridWidthMutator {

		/**
		 * Sets the grid width for the grid width property.
		 * 
		 * @param aWidth The grid width to be stored by the grid width property.
		 */
		void setGridWidth( int aWidth );
	}

	/**
	 * Provides a builder method for a grid width property returning the builder
	 * for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface GridWidthBuilder<B extends GridWidthBuilder<B>> {

		/**
		 * Sets the grid width for the grid width property.
		 * 
		 * @param aWidth The grid width to be stored by the grid width property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withGridWidth( int aWidth );
	}

	/**
	 * Provides a grid width property.
	 */
	public interface GridWidthProperty extends GridWidthAccessor, GridWidthMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given integer (setter) as of
		 * {@link #setGridWidth(int)} and returns the very same value (getter).
		 * 
		 * @param aWidth The integer to set (via {@link #setGridWidth(int)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default int letGridWidth( int aWidth ) {
			setGridWidth( aWidth );
			return aWidth;
		}
	}
}
