// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

import org.refcodes.data.Encoding;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;

/**
 * Implementation of the {@link InputStreamStringBuilder} interface. The
 * {@link InputStream} is not touched earlier than the beginning of the
 * conversion operation. If no encoding is set, then UTF-8 is used by default.
 */
public class InputStreamStringBuilderImpl implements InputStreamStringBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private InputStream _inputStream = null;

	private String _lines = null;

	private String _encoding = Encoding.UTF_8.getCode();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getEncoding() {
		return _encoding;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEncoding( String aEncoding ) {
		_encoding = aEncoding;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InputStream getInputStream() {
		return _inputStream;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setInputStream( InputStream aInputStream ) {
		_inputStream = aInputStream;
		_lines = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString( InputStream aInputStream, String aEncoding ) throws IOException {
		if ( aInputStream == null ) return null;
		try (Scanner theScanner = new Scanner( aInputStream, aEncoding )) {
			theScanner.useDelimiter( "\\A" );
			return theScanner.hasNext() ? theScanner.next() : "";
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		try {
			return toString( _encoding );
		}
		catch ( IOException e ) {
			throw new HiddenException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString( String aEncoding ) throws IOException {
		if ( _lines == null ) {
			_lines = toString( _inputStream, aEncoding );
		}
		return _lines;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] toStrings() throws IOException {
		if ( _lines == null ) {
			_lines = toString( _inputStream, _encoding );
		}
		return _lines.split( "\\r\\n|\\n|\\r" );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
